<?php
/**
 * @package         FirePlugins Framework
 * @version         1.1.99
 * 
 * @author          FirePlugins <info@fireplugins.com>
 * @link            https://www.fireplugins.com
 * @copyright       Copyright © 2024 FirePlugins All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
*/

namespace FPFramework\Libs;

if (!defined('ABSPATH'))
{
	exit; // Exit if accessed directly.
}

class Translations
{
	/**
	 * Holds all translations of the framework and the current plugin.
	 * 
	 * @var  array
	 */
	private $translations = [];

	/**
	 * Stores cached translations.
	 * 
	 * @var  array
	 */
	private $cached = [];

	public function __construct()
	{
		$this->translations = $this->getTranslations();
	}

	/**
	 * Retrieves the translation of $text
	 * 
	 * @param   String  $text
	 * @param   String  $fallback
	 * 
	 * @return  String
	 */
	public function _($text, $fallback = null)
	{
		if (is_numeric($text))
		{
			return $text;
		}
		
		if (!is_string($text) && !is_numeric($text))
		{
			return '';
		}

		if (empty($text))
		{
			return '';
		}
		
		if (isset($this->cached[$text]))
		{
			return $this->cached[$text];
		}

		if ($fallback && isset($this->cached[$fallback]))
		{
			return $this->cached[$fallback];
		}

		if ($translation = $this->retrieve($text, $fallback))
		{
			$this->cached[$translation['source']] = $translation['value'];

			return $translation['value'];
		}

		return $fallback ? trim($fallback) : trim($text);
	}

	/**
	 * Retrieves translation of given text or of fallback text.
	 * If none found, returns false
	 * 
	 * @param   string  $text
	 * @param   string  $fallback
	 * 
	 * @return  mixed
	 */
	private function retrieve($text, $fallback = '')
	{
		if (!is_string($text) && !is_numeric($text))
		{
			return '';
		}

		$translationOfText = $this->findText($text);
		if ($translationOfText !== false)
		{
			return [
				'source' => $text,
				'value' => $translationOfText
			];
		}

		$fallback = !empty($fallback) ? $fallback : $text;

		$translationOfFallback = $this->findText($fallback);
		if ($translationOfFallback !== false)
		{
			return [
				'source' => $fallback,
				'value' => $translationOfFallback
			];
		}

		return false;
	}

	/**
	 * Tries to find translation of text. Returns false if fails.
	 * 
	 * @param   string  $text
	 * 
	 * @return  mixed
	 */
	private function findText($text)
	{
		return isset($this->translations[strtoupper(trim($text))]) ? $this->translations[strtoupper(trim($text))] : false;
	}

	/**
	 * All Translations
	 * 
	 * @return array
	 */
	public static function getTranslations()
	{
		return [
			'FPF_NAME' => 'FirePlugins',
			'FPF_DOMAIN' => 'Fireplugins.com',
			'FPF_DASHBOARD' => __('Dashboard', 'fpf-framework'),
			'FPF_OVERVIEW' => __('Overview', 'fpf-framework'),
			'FPF_DOCUMENTATION' => __('Documentation', 'fpf-framework'),
			'FPF_FIREBOX' => __('FireBox', 'fpf-framework'),
			'FPF_FIREBOX_SHORT' => __('FB', 'fpf-framework'),
			'FPF_FIRESCHEMA' => __('FireSchema', 'fpf-framework'),
			'FPF_FIREFORMS' => __('FireForms', 'fpf-framework'),
			'FPF_DOCS' => __('Docs', 'fpf-framework'),
			'FPF_NEW' => __('New', 'fpf-framework'),
			'FPF_HELP' => __('Help', 'fpf-framework'),
			'FPF_ID' => __('ID', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS' => __('Display Conditions', 'fpf-framework'),
			'FPF_MODE' => __('Mode', 'fpf-framework'),
			'FPF_IMPORT' => __('Import', 'fpf-framework'),
			'FPF_SETTINGS' => __('Settings', 'fpf-framework'),
			'FPF_DISABLED' => __('Disabled', 'fpf-framework'),
			'FPF_INCLUDE' => __('Include', 'fpf-framework'),
			'FPF_EXCLUDE' => __('Exclude', 'fpf-framework'),
			'FPF_ENABLED' => __('Enabled', 'fpf-framework'),
			'FPF_SESSION' => __('Session', 'fpf-framework'),
			'FPF_ONCE_PER_SESSION' => __('Once Per Session', 'fpf-framework'),
			'FPF_ONCE_PER_DAY' => __('Once Per Day', 'fpf-framework'),
			'FPF_ONCE_PER_WEEK' => __('Once Per Week', 'fpf-framework'),
			'FPF_PER_SESSION' => __('Per Session', 'fpf-framework'),
			'FPF_PER_DAY' => __('Per Day', 'fpf-framework'),
			'FPF_PER_WEEK' => __('Per Week', 'fpf-framework'),
			'FPF_PER_MONTH' => __('Per Month', 'fpf-framework'),
			'FPF_FOREVER' => __('Forever', 'fpf-framework'),
			'FPF_IMPRESSIONS' => __('Impressions', 'fpf-framework'),
			'FPF_WIDTH' => __('Width', 'fpf-framework'),
			'FPF_HEIGHT' => __('Height', 'fpf-framework'),
			'FPF_PADDING' => __('Padding', 'fpf-framework'),
			'FPF_MARGIN' => __('Margin', 'fpf-framework'),
			'FPF_COLOR' => __('Color', 'fpf-framework'),
			'FPF_TEXT_COLOR' => __('Text Color', 'fpf-framework'),
			'FPF_BG_COLOR' => __('Background Color', 'fpf-framework'),
			'FPF_NONE' => __('None', 'fpf-framework'),
			'FPF_BORDER' => __('Border', 'fpf-framework'),
			'FPF_BORDER_DESC' => __('Set the border style, color, width and radius.', 'fpf-framework'),
			'FPF_STYLE' => __('Style', 'fpf-framework'),
			'FPF_SOLID' => __('Solid', 'fpf-framework'),
			'FPF_DOTTED' => __('Dotted', 'fpf-framework'),
			'FPF_DOUBLE' => __('Double', 'fpf-framework'),
			'FPF_DASHED' => __('Dashed', 'fpf-framework'),
			'FPF_INSET' => __('Inset', 'fpf-framework'),
			'FPF_OUTSET' => __('Outset', 'fpf-framework'),
			'FPF_GROOVE' => __('Groove', 'fpf-framework'),
			'FPF_RIDGE' => __('Ridge', 'fpf-framework'),
			'FPF_HIDDEN' => __('Hidden', 'fpf-framework'),
			'FPF_HIDE' => __('Hide', 'fpf-framework'),
			'FPF_ENABLED' => __('Enabled', 'fpf-framework'),
			'FPF_ICON' => __('Icon', 'fpf-framework'),
			'FPF_IMAGE' => __('Image', 'fpf-framework'),
			'FPF_BG_IMAGE' => __('Background Image', 'fpf-framework'),
			'FPF_BG_IMAGE_DESC' => __('Set the background image settings, repeat, size and position.', 'fpf-framework'),
			'FPF_REPEAT' => __('Repeat', 'fpf-framework'),
			'FPF_SIZE' => __('Size', 'fpf-framework'),
			'FPF_AUTO' => __('Auto', 'fpf-framework'),
			'FPF_POSITION' => __('Position', 'fpf-framework'),
			'FPF_FONT_SIZE' => __('Font Size', 'fpf-framework'),
			'FPF_FONT_SIZE_DESC' => __('Set the font size.', 'fpf-framework'),
			'FPF_SELECT_IMAGE' => __('Select Image', 'fpf-framework'),
			'FPF_ALL' => __('All', 'fpf-framework'),
			'FPF_ANY' => __('Any', 'fpf-framework'),
			'FPF_POST' => __('Post', 'fpf-framework'),
			'FPF_PAGE' => __('Page', 'fpf-framework'),
			'FPF_PATHS' => __('Paths', 'fpf-framework'),
			'FPF_CPT' => __('Custom Post Type', 'fpf-framework'),
			'FPF_CPTS' => __('Custom Post Types', 'fpf-framework'),
			'FPF_POST_TAG' => __('Post Tag', 'fpf-framework'),
			'FPF_TAG' => __('Tag', 'fpf-framework'),
			'FPF_TAGS' => __('Tags', 'fpf-framework'),
			'FPF_POST_CATEGORY' => __('Post Category', 'fpf-framework'),
			'FPF_CATEGORIES' => __('Categories', 'fpf-framework'),
			'FPF_ADVANCED' => __('Advanced', 'fpf-framework'),
			'FPF_PA_ADVANCED' => __('Advanced visitor targeting', 'fpf-framework'),
			'FPF_CUSTOM_CODE' => __('Custom Code', 'fpf-framework'),
			'FPF_CUSTOM_CODE_DESC' => __('Enter your own Custom CSS & JavaScript in this section.', 'fpf-framework'),
			'FPF_CUSTOM_JAVASCRIPT' => __('Custom JavaScript', 'fpf-framework'),
			'FPF_CUSTOM_JAVASCRIPT_DESC' => __('Write your own custom JavaScript code (You don\'t need to enter the &lt;script&gt;&lt;/script&gt; tags).', 'fpf-framework'),
			'FPF_CUSTOM_CSS' => __('Custom CSS', 'fpf-framework'),
			'FPF_CUSTOM_CSS_DESC' => __('Write your own custom CSS Code (You don\'t need to enter the &lt;style&gt;&lt;/style&gt; tags).', 'fpf-framework'),
			'FPF_USE_GLOBAL' => __('Use Global', 'fpf-framework'),
			'FPF_YES' => __('Yes', 'fpf-framework'),
			'FPF_NO' => __('No', 'fpf-framework'),
			'FPF_ONLY' => __('Only', 'fpf-framework'),
			'FPF_CLASS_SUFFIX' => __('Class Suffix', 'fpf-framework'),
			'FPF_CLASS_SUFFIX_DESC' => __('Enter a class suffix.', 'fpf-framework'),
			'FPF_ALL_RIGHTS_RESERVED' => __('All Rights Reserved', 'fpf-framework'),
			'FPF_NEED_SUPPORT' => __('Need support?', 'fpf-framework'),
			'FPF_SUPPORT' => __('Support', 'fpf-framework'),
			'FPF_READ_THE_DOCS' => __('Read the Documentation', 'fpf-framework'),
			'FPF_OR' => __('or', 'fpf-framework'),
			'FPF_DROP_EMAIL' => __('Drop me an e-mail', 'fpf-framework'),
			'FPF_UPLOAD' => __('Upload', 'fpf-framework'),
			'FPF_PREVIEW' => __('Preview', 'fpf-framework'),
			'FPF_MEDIA_UPLOAD_TMP_IMG_ALT' => __('media uploader preview image', 'fpf-framework'),
			'FPF_ENSURE_PLUGIN_FOLDER_IS_READABLE' => __('Please make sure that the plugins folder is readable.', 'fpf-framework'),
			'FPF_PA_MENU_ITEM_AS' => __('Menu Item', 'fpf-framework'),
			'FPF_PA_MENU_ITEM_AS_DESC' => __('Target visitors who are browsing specific menu items', 'fpf-framework'),
			'FPF_PA_DATETIME' => __('Datetime', 'fpf-framework'),
			'FPF_PA_DATETIME_DESC' => __('Target visitors based on your server\'s datetime', 'fpf-framework'),
			'FPF_SELECTION' => __('Selection', 'fpf-framework'),
			'FPF_PA_ASSIGN_MENU_CHILD' => __('Also on child items', 'fpf-framework'),
			'FPF_PA_ASSIGN_MENU_CHILD_DESC' => __('Also assign to child items of the selected items?', 'fpf-framework'),
			'FPF_URL' => __('URL', 'fpf-framework'),
			'FPF_PA_URL_DESC' => __('Target visitors who are browsing specific URLs', 'fpf-framework'),
			'FPF_REGUAL_EXPRESSION' => __('Use Regular Expression', 'fpf-framework'),
			'FPF_REGUAL_EXPRESSION_DESC' => __('Select to treat the value as regular expressions.', 'fpf-framework'),
			'FPF_PA_MENU_ITEM_HINT' => __('Start searching for menu items.', 'fpf-framework'),
			'FPF_DATETIME_START' => __('Start Datetime', 'fpf-framework'),
			'FPF_DATETIME_START_DESC' => __('Enter the date to start publishing', 'fpf-framework'),
			'FPF_DATETIME_END' => __('End Datetime', 'fpf-framework'),
			'FPF_DATETIME_END_DESC' => __('Enter the date to end publishing', 'fpf-framework'),
			'FPF_PA_TIMERANGE' => __('Time Range', 'fpf-framework'),
			'FPF_PA_TIMERANGE_DESC' => __('Target visitors based on your server\'s time', 'fpf-framework'),
			'FPF_TIMEPICKER_START' => __('Start Time', 'fpf-framework'),
			'FPF_TIMEPICKER_START_DESC' => __('Enter the time to start publishing', 'fpf-framework'),
			'FPF_TIMEPICKER_END' => __('End Time', 'fpf-framework'),
			'FPF_TIMEPICKER_END_DESC' => __('Enter the time to end publishing', 'fpf-framework'),
			'FPF_USER_GROUP' => __('User Group', 'fpf-framework'),
			'FPF_USER_GROUP_HINT' => __('Start searching for a user group', 'fpf-framework'),
			'FPF_PA_PAGE_URL_AS_DESC' => __('Select menu items.', 'fpf-framework'),
			'FPF_USER_ID' => __('User ID', 'fpf-framework'),
			'FPF_USER_ID_DESC' => __('Target specific WordPress Users by their IDs', 'fpf-framework'),
			'FPF_NUMBER_PAGEVIEWS' => __('Number of Pageviews', 'fpf-framework'),
			'FPF_NUMBER_PAGEVIEWS_DESC' => __('Target visitors who have viewed certain number of pages', 'fpf-framework'),
			'FPF_EXACTLY' => __('Exactly', 'fpf-framework'),
			'FPF_FEWER' => __('Fewer', 'fpf-framework'),
			'FPF_FEWER_THAN' => __('Fewer than', 'fpf-framework'),
			'FPF_FEWER_THAN_OR_EQUAL_TO' => __('Fewer than or equal to', 'fpf-framework'),
			'FPF_LESS_THAN' => __('Less than', 'fpf-framework'),
			'FPF_LESS_THAN_OR_EQUAL_TO' => __('Less than or equal to', 'fpf-framework'),
			'FPF_GREATER' => __('Greater', 'fpf-framework'),
			'FPF_GREATER_THAN' => __('Greater than', 'fpf-framework'),
			'FPF_GREATER_THAN_OR_EQUAL_TO' => __('Greater than or equal to', 'fpf-framework'),
			'FPF_MORE_THAN' => __('More than', 'fpf-framework'),
			'FPF_MORE_THAN_OR_EQUAL_TO' => __('More than or equal to', 'fpf-framework'),
			'FPF_MATCH' => __('Match', 'fpf-framework'),
			'FPF_PAGEVIEWS' => __('Pageviews', 'fpf-framework'),
			'FPF_TIMEONSITE' => __('Time on Site', 'fpf-framework'),
			'FPF_TIMEONSITE_DESC' => __('Target visitors who have spent a specific amount of time on your site.', 'fpf-framework'),
			'FPF_NO_ITEMS_FOUND' => __('No Items found.', 'fpf-framework'),
			'FPF_TYPE' => __('Type', 'fpf-framework'),
			'FPF_POST_HINT' => __('Start searching for a post', 'fpf-framework'),
			'FPF_PAGE_HINT' => __('Start searching for a page', 'fpf-framework'),
			'FPF_PA_USER_ROLES_HINT' => __('Start searching for user roles', 'fpf-framework'),
			'FPF_PA_CPTS_DESC' => __('Target specific custom post types', 'fpf-framework'),
			'FPF_PA_CPTS_HINT' => __('Start searching for a custom post type', 'fpf-framework'),
			'FPF_POST_TAG_HINT' => __('Start searching for a post tag', 'fpf-framework'),
			'FPF_PA_POST_CATEGORIES_DESC' => __('Target specific post categories', 'fpf-framework'),
			'FPF_PA_POST_CATEGORIES_HINT' => __('Start searching for post categories.', 'fpf-framework'),
			'FPF_REFERRER_URL' => __('Referrer URL', 'fpf-framework'),
			'FPF_REFERRER_URL_DESC' => __('Target visitors who land on your site from a specific traffic source', 'fpf-framework'),
			'FPF_REFERRER_URL_SELECTION_DESC' => __('Enter one Referrer URL per line. Example:<br>google.com<br>facebook.com/mypage', 'fpf-framework'),
			'FPF_REFERRER_URL_DESC' => __('Keep in mind that URL Referrer discovery is not always 100% accurate. Some servers may use proxies that strip this information out and it can be easily forged.', 'fpf-framework'),
			'FPF_IP_ADDRESS' => __('IP Address', 'fpf-framework'),
			'FPF_IP_ADDRESS_DESC' => __('Target visitors who are behind a specific IP address (range)', 'fpf-framework'),
			'FPF_IP_ADDRESS_SELECTION_DESC' => __('Enter a list of comma and/or \'enter\' separated ip addresses and ranges. Example:<br /> 127.0.0.1,<br /> 192.10-120.2,<br /> 168', 'fpf-framework'),
			'FPF_PHP' => __('PHP', 'fpf-framework'),
			'FPF_PHP_DESC' => __('Target visitors evaluating custom PHP code.', 'fpf-framework'),
			'FPF_PHP_CODE' => __('PHP Code', 'fpf-framework'),
			'FPF_PHP_SELECTION_DESC' => __('Enter a piece of PHP code to evaluate.  The code must return the value true or false. Example: <Br>$current_user = wp_get_current_user();<br>return ($current_user->display_name == \'John_Doe\');<br><br>A collection of ready-to-use PHP assignment scripts is available. <a target="_blank" href="https://www.fireplugins.com/docs/general/extending-general/custom-php-assignments/">View collection</a>', 'fpf-framework'),
			'FPF_COOKIE' => __('Cookie', 'fpf-framework'),
			'FPF_PA_COOKIE_DESC' => __('Target visitors who have specific cookies stored in their browser', 'fpf-framework'),
			'FPF_COOKIE_NAME' => __('Cookie Name', 'fpf-framework'),
			'FPF_PA_COOKIE_NAME_DESC' => __('Enter the name of the cookie to assign to', 'fpf-framework'),
			'FPF_MATCH' => __('Match', 'fpf-framework'),
			'FPF_PA_MATCH_DESC' => __('The used matching method to compare the value', 'fpf-framework'),
			'FPF_EXISTS' => __('Exists', 'fpf-framework'),
			'FPF_NOT_EXISTS' => __('Does not exists', 'fpf-framework'),
			'FPF_IS_EQUAL' => __('Equals', 'fpf-framework'),
			'FPF_DOES_NOT_EQUAL' => __('Does not equal', 'fpf-framework'),
			'FPF_CONTAINS' => __('Contains', 'fpf-framework'),
			'FPF_DOES_NOT_CONTAIN' => __('Does not contain', 'fpf-framework'),
			'FPF_STARTS_WITH' => __('Starts With', 'fpf-framework'),
			'FPF_DOES_NOT_START_WITH' => __('Does not start with', 'fpf-framework'),
			'FPF_ENDS_WITH' => __('Ends With', 'fpf-framework'),
			'FPF_DOES_NOT_END_WITH' => __('Does not end with', 'fpf-framework'),
			'FPF_COOKIE_CONTENT' => __('Cookie Content', 'fpf-framework'),
			'FPF_PA_COOKIE_NAME_PARAM_CONTENT_DESC' => __('The cookie\'s content', 'fpf-framework'),
			'FPF_METABOX_PA_VISITOR_PLATFORM' => __('Visitor Platform', 'fpf-framework'),
			'FPF_METABOX_PA_VISITOR_PLATFORM_DESC' => __('Target visitors who are using Mobile, Google Chrome, or even Windows', 'fpf-framework'),
			'FPF_PA_DEVICES_DESC2' => __('Keep in mind that device detection is not always 100% accurate. Users can setup their browser to mimic other devices.', 'fpf-framework'),
			'FPF_DEVICE' => __('Device', 'fpf-framework'),
			'FPF_DEVICES' => __('Devices', 'fpf-framework'),
			'FPF_PA_DEVICE_DESC' => __('Target visitors using specific device such as Mobile, Tablet or Desktop.', 'fpf-framework'),
			'FPF_PA_DEVICES_HINT' => __('Start searching for devices.', 'fpf-framework'),
			'FPF_DESKTOP' => __('Desktop', 'fpf-framework'),
			'FPF_TABLET' => __('Tablet', 'fpf-framework'),
			'FPF_MOBILE' => __('Mobile', 'fpf-framework'),
			'FPF_X_DEVICE_SETTINGS' => __('%s device settings', 'fpf-frameowrk'),
			'FPF_CHROME' => __('Chrome', 'fpf-framework'),
			'FPF_FIREFOX' => __('Firefox', 'fpf-framework'),
			'FPF_EDGE' => __('Edge', 'fpf-framework'),
			'FPF_IE' => __('Internet Explorer', 'fpf-framework'),
			'FPF_SAFARI' => __('Safari', 'fpf-framework'),
			'FPF_OPERA' => __('Opera', 'fpf-framework'),
			'FPF_BROWSER' => __('Browser', 'fpf-framework'),
			'FPF_PA_BROWSER_DESC' => __('Target visitors who are browsing your site with specific browsers such as Chrome, Firefox or Internet Explorer', 'fpf-framework'),
			'FPF_PA_BROWSERS_AS_DESC' => __('Select the browsers to assign to', 'fpf-framework'),
			'FPF_PA_BROWSERS_HINT' => __('Start searching for browsers.', 'fpf-framework'),
			'FPF_OS' => __('Operating System', 'fpf-framework'),
			'FPF_PA_OS_DESC' => __('Target visitors who are using specific operating systems such as Windows, Linux or Mac', 'fpf-framework'),
			'FPF_PA_OSES_AS_DESC' => __('Select the operating systems to assign to', 'fpf-framework'),
			'FPF_PA_OSES_HINT' => __('Start searching for operating systems.', 'fpf-framework'),
			'FPF_LINUX' => __('Linux', 'fpf-framework'),
			'FPF_MAC' => __('MacOS', 'fpf-framework'),
			'FPF_ANDROID' => __('Android', 'fpf-framework'),
			'FPF_IOS' => __('iOS', 'fpf-framework'),
			'FPF_WINDOWS' => __('Windows', 'fpf-framework'),
			'FPF_BLACKBERRY' => __('Blackberry', 'fpf-framework'),
			'FPF_CHROMEOS' => __('Chrome OS', 'fpf-framework'),
			'FPF_PA_GEOLOCATING' => __('Geolocating', 'fpf-framework'),
			'FPF_PA_GEOLOCATING_DESC' => __('Target visitors who are physically in a specific region', 'fpf-framework'),
			'FPF_CONTINENT' => __('Continent', 'fpf-framework'),
			'FPF_CONTINENT_DESC' => __('Target visitors who are physically in a specific continent', 'fpf-framework'),
			'FPF_PA_CONTINENT_HINT' => __('Start searching for continents.', 'fpf-framework'),
			'FPF_PA_CONTINENT_AS_DESC' => __('Select the continents to assign to.', 'fpf-framework'),
			'FPF_CONTINENT_AF' => __('Africa', 'fpf-framework'),
			'FPF_CONTINENT_AS' => __('Asia', 'fpf-framework'),
			'FPF_CONTINENT_EU' => __('Europe', 'fpf-framework'),
			'FPF_CONTINENT_NA' => __('North America', 'fpf-framework'),
			'FPF_CONTINENT_SA' => __('South America', 'fpf-framework'),
			'FPF_CONTINENT_OC' => __('Oceania', 'fpf-framework'),
			'FPF_CONTINENT_AN' => __('Antartica', 'fpf-framework'),
			'FPF_COUNTRY' => __('Country', 'fpf-framework'),
			'FPF_COUNTRIES' => __('Countries', 'fpf-framework'),
			'FPF_COUNTRY_CODE' => __('Country Code', 'fpf-framework'),
			'FPF_CITY' => __('City', 'fpf-framework'),
			'FPF_CITY_DESC' => __('Target visitors who are physically in a specific city.', 'fpf-framework'),
			'FPF_COUNTRY_DESC' => __('Target visitors who are physically in a specific country.', 'fpf-framework'),
			'FPF_PA_COUNTRY_AS_DESC' => __('Select the countries to assign to', 'fpf-framework'),
			'FPF_PA_COUNTRY_HINT' => __('Start searching for countries.', 'fpf-framework'),
			'FPF_COUNTRY_AF' => __('Afghanistan', 'fpf-framework'),
			'FPF_COUNTRY_AX' => __('Aland Islands', 'fpf-framework'),
			'FPF_COUNTRY_AL' => __('Albania', 'fpf-framework'),
			'FPF_COUNTRY_DZ' => __('Algeria', 'fpf-framework'),
			'FPF_COUNTRY_AS' => __('American Samoa', 'fpf-framework'),
			'FPF_COUNTRY_AD' => __('Andorra', 'fpf-framework'),
			'FPF_COUNTRY_AO' => __('Angola', 'fpf-framework'),
			'FPF_COUNTRY_AI' => __('Anguilla', 'fpf-framework'),
			'FPF_COUNTRY_AQ' => __('Antarctica', 'fpf-framework'),
			'FPF_COUNTRY_AG' => __('Antigua and Barbuda', 'fpf-framework'),
			'FPF_COUNTRY_AR' => __('Argentina', 'fpf-framework'),
			'FPF_COUNTRY_AM' => __('Armenia', 'fpf-framework'),
			'FPF_COUNTRY_AW' => __('Aruba', 'fpf-framework'),
			'FPF_COUNTRY_AU' => __('Australia', 'fpf-framework'),
			'FPF_COUNTRY_AT' => __('Austria', 'fpf-framework'),
			'FPF_COUNTRY_AZ' => __('Azerbaijan', 'fpf-framework'),
			'FPF_COUNTRY_BS' => __('Bahamas', 'fpf-framework'),
			'FPF_COUNTRY_BH' => __('Bahrain', 'fpf-framework'),
			'FPF_COUNTRY_BD' => __('Bangladesh', 'fpf-framework'),
			'FPF_COUNTRY_BB' => __('Barbados', 'fpf-framework'),
			'FPF_COUNTRY_BY' => __('Belarus', 'fpf-framework'),
			'FPF_COUNTRY_BE' => __('Belgium', 'fpf-framework'),
			'FPF_COUNTRY_BZ' => __('Belize', 'fpf-framework'),
			'FPF_COUNTRY_BJ' => __('Benin', 'fpf-framework'),
			'FPF_COUNTRY_BM' => __('Bermuda', 'fpf-framework'),
			'FPF_COUNTRY_BQ_BO' => __('Bonaire', 'fpf-framework'),
			'FPF_COUNTRY_BQ_SA' => __('Saba', 'fpf-framework'),
			'FPF_COUNTRY_BQ_SE' => __('Sint Eustatius', 'fpf-framework'),
			'FPF_COUNTRY_BT' => __('Bhutan', 'fpf-framework'),
			'FPF_COUNTRY_BO' => __('Bolivia', 'fpf-framework'),
			'FPF_COUNTRY_BA' => __('Bosnia and Herzegovina', 'fpf-framework'),
			'FPF_COUNTRY_BW' => __('Botswana', 'fpf-framework'),
			'FPF_COUNTRY_BV' => __('Bouvet Island', 'fpf-framework'),
			'FPF_COUNTRY_BR' => __('Brazil', 'fpf-framework'),
			'FPF_COUNTRY_IO' => __('British Indian Ocean Territory', 'fpf-framework'),
			'FPF_COUNTRY_BN' => __('Brunei Darussalam', 'fpf-framework'),
			'FPF_COUNTRY_BG' => __('Bulgaria', 'fpf-framework'),
			'FPF_COUNTRY_BF' => __('Burkina Faso', 'fpf-framework'),
			'FPF_COUNTRY_BI' => __('Burundi', 'fpf-framework'),
			'FPF_COUNTRY_KH' => __('Cambodia', 'fpf-framework'),
			'FPF_COUNTRY_CM' => __('Cameroon', 'fpf-framework'),
			'FPF_COUNTRY_CA' => __('Canada', 'fpf-framework'),
			'FPF_COUNTRY_CV' => __('Cape Verde', 'fpf-framework'),
			'FPF_COUNTRY_KY' => __('Cayman Islands', 'fpf-framework'),
			'FPF_COUNTRY_CF' => __('Central African Republic', 'fpf-framework'),
			'FPF_COUNTRY_TD' => __('Chad', 'fpf-framework'),
			'FPF_COUNTRY_CL' => __('Chile', 'fpf-framework'),
			'FPF_COUNTRY_CN' => __('China', 'fpf-framework'),
			'FPF_COUNTRY_CX' => __('Christmas Island', 'fpf-framework'),
			'FPF_COUNTRY_CC' => __('Cocos (Keeling) Islands', 'fpf-framework'),
			'FPF_COUNTRY_CO' => __('Colombia', 'fpf-framework'),
			'FPF_COUNTRY_KM' => __('Comoros', 'fpf-framework'),
			'FPF_COUNTRY_CG' => __('Congo', 'fpf-framework'),
			'FPF_COUNTRY_CD' => __('Congo, The Democratic Republic of the', 'fpf-framework'),
			'FPF_COUNTRY_CK' => __('Cook Islands', 'fpf-framework'),
			'FPF_COUNTRY_CR' => __('Costa Rica', 'fpf-framework'),
			'FPF_COUNTRY_CI' => __('Cote d\'Ivoire', 'fpf-framework'),
			'FPF_COUNTRY_HR' => __('Croatia', 'fpf-framework'),
			'FPF_COUNTRY_CU' => __('Cuba', 'fpf-framework'),
			'FPF_COUNTRY_CW' => __('Curaçao', 'fpf-framework'),
			'FPF_COUNTRY_CY' => __('Cyprus', 'fpf-framework'),
			'FPF_COUNTRY_CZ' => __('Czech Republic', 'fpf-framework'),
			'FPF_COUNTRY_DK' => __('Denmark', 'fpf-framework'),
			'FPF_COUNTRY_DJ' => __('Djibouti', 'fpf-framework'),
			'FPF_COUNTRY_DM' => __('Dominica', 'fpf-framework'),
			'FPF_COUNTRY_DO' => __('Dominican Republic', 'fpf-framework'),
			'FPF_COUNTRY_EC' => __('Ecuador', 'fpf-framework'),
			'FPF_COUNTRY_EG' => __('Egypt', 'fpf-framework'),
			'FPF_COUNTRY_SV' => __('El Salvador', 'fpf-framework'),
			'FPF_COUNTRY_GQ' => __('Equatorial Guinea', 'fpf-framework'),
			'FPF_COUNTRY_ER' => __('Eritrea', 'fpf-framework'),
			'FPF_COUNTRY_EE' => __('Estonia', 'fpf-framework'),
			'FPF_COUNTRY_ET' => __('Ethiopia', 'fpf-framework'),
			'FPF_COUNTRY_FK' => __('Falkland Islands (Malvinas)', 'fpf-framework'),
			'FPF_COUNTRY_FO' => __('Faroe Islands', 'fpf-framework'),
			'FPF_COUNTRY_FJ' => __('Fiji', 'fpf-framework'),
			'FPF_COUNTRY_FI' => __('Finland', 'fpf-framework'),
			'FPF_COUNTRY_FR' => __('France', 'fpf-framework'),
			'FPF_COUNTRY_GF' => __('French Guiana', 'fpf-framework'),
			'FPF_COUNTRY_PF' => __('French Polynesia', 'fpf-framework'),
			'FPF_COUNTRY_TF' => __('French Southern Territories', 'fpf-framework'),
			'FPF_COUNTRY_GA' => __('Gabon', 'fpf-framework'),
			'FPF_COUNTRY_GM' => __('Gambia', 'fpf-framework'),
			'FPF_COUNTRY_GE' => __('Georgia', 'fpf-framework'),
			'FPF_COUNTRY_DE' => __('Germany', 'fpf-framework'),
			'FPF_COUNTRY_GH' => __('Ghana', 'fpf-framework'),
			'FPF_COUNTRY_GI' => __('Gibraltar', 'fpf-framework'),
			'FPF_COUNTRY_GR' => __('Greece', 'fpf-framework'),
			'FPF_COUNTRY_GL' => __('Greenland', 'fpf-framework'),
			'FPF_COUNTRY_GD' => __('Grenada', 'fpf-framework'),
			'FPF_COUNTRY_GP' => __('Guadeloupe', 'fpf-framework'),
			'FPF_COUNTRY_GU' => __('Guam', 'fpf-framework'),
			'FPF_COUNTRY_GT' => __('Guatemala', 'fpf-framework'),
			'FPF_COUNTRY_GG' => __('Guernsey', 'fpf-framework'),
			'FPF_COUNTRY_GN' => __('Guinea', 'fpf-framework'),
			'FPF_COUNTRY_GW' => __('Guinea-Bissau', 'fpf-framework'),
			'FPF_COUNTRY_GY' => __('Guyana', 'fpf-framework'),
			'FPF_COUNTRY_HT' => __('Haiti', 'fpf-framework'),
			'FPF_COUNTRY_HM' => __('Heard Island and McDonald Islands', 'fpf-framework'),
			'FPF_COUNTRY_VA' => __('Holy See (Vatican City State)', 'fpf-framework'),
			'FPF_COUNTRY_HN' => __('Honduras', 'fpf-framework'),
			'FPF_COUNTRY_HK' => __('Hong Kong', 'fpf-framework'),
			'FPF_COUNTRY_HU' => __('Hungary', 'fpf-framework'),
			'FPF_COUNTRY_IS' => __('Iceland', 'fpf-framework'),
			'FPF_COUNTRY_IN' => __('India', 'fpf-framework'),
			'FPF_COUNTRY_ID' => __('Indonesia', 'fpf-framework'),
			'FPF_COUNTRY_IR' => __('Iran, Islamic Republic of', 'fpf-framework'),
			'FPF_COUNTRY_IQ' => __('Iraq', 'fpf-framework'),
			'FPF_COUNTRY_IE' => __('Ireland', 'fpf-framework'),
			'FPF_COUNTRY_IM' => __('Isle of Man', 'fpf-framework'),
			'FPF_COUNTRY_IL' => __('Israel', 'fpf-framework'),
			'FPF_COUNTRY_IT' => __('Italy', 'fpf-framework'),
			'FPF_COUNTRY_JM' => __('Jamaica', 'fpf-framework'),
			'FPF_COUNTRY_JP' => __('Japan', 'fpf-framework'),
			'FPF_COUNTRY_JE' => __('Jersey', 'fpf-framework'),
			'FPF_COUNTRY_JO' => __('Jordan', 'fpf-framework'),
			'FPF_COUNTRY_KZ' => __('Kazakhstan', 'fpf-framework'),
			'FPF_COUNTRY_KE' => __('Kenya', 'fpf-framework'),
			'FPF_COUNTRY_KI' => __('Kiribati', 'fpf-framework'),
			'FPF_COUNTRY_KP' => __('Korea, Democratic People\'s Republic of', 'fpf-framework'),
			'FPF_COUNTRY_KR' => __('Korea, Republic of', 'fpf-framework'),
			'FPF_COUNTRY_KW' => __('Kuwait', 'fpf-framework'),
			'FPF_COUNTRY_KG' => __('Kyrgyzstan', 'fpf-framework'),
			'FPF_COUNTRY_LA' => __('Lao People\'s Democratic Republic', 'fpf-framework'),
			'FPF_COUNTRY_LV' => __('Latvia', 'fpf-framework'),
			'FPF_COUNTRY_LB' => __('Lebanon', 'fpf-framework'),
			'FPF_COUNTRY_LS' => __('Lesotho', 'fpf-framework'),
			'FPF_COUNTRY_LR' => __('Liberia', 'fpf-framework'),
			'FPF_COUNTRY_LY' => __('Libyan Arab Jamahiriya', 'fpf-framework'),
			'FPF_COUNTRY_LI' => __('Liechtenstein', 'fpf-framework'),
			'FPF_COUNTRY_LT' => __('Lithuania', 'fpf-framework'),
			'FPF_COUNTRY_LU' => __('Luxembourg', 'fpf-framework'),
			'FPF_COUNTRY_MO' => __('Macao', 'fpf-framework'),
			'FPF_COUNTRY_MK' => __('Macedonia', 'fpf-framework'),
			'FPF_COUNTRY_MG' => __('Madagascar', 'fpf-framework'),
			'FPF_COUNTRY_MW' => __('Malawi', 'fpf-framework'),
			'FPF_COUNTRY_MY' => __('Malaysia', 'fpf-framework'),
			'FPF_COUNTRY_MV' => __('Maldives', 'fpf-framework'),
			'FPF_COUNTRY_ML' => __('Mali', 'fpf-framework'),
			'FPF_COUNTRY_MT' => __('Malta', 'fpf-framework'),
			'FPF_COUNTRY_MH' => __('Marshall Islands', 'fpf-framework'),
			'FPF_COUNTRY_MQ' => __('Martinique', 'fpf-framework'),
			'FPF_COUNTRY_MR' => __('Mauritania', 'fpf-framework'),
			'FPF_COUNTRY_MU' => __('Mauritius', 'fpf-framework'),
			'FPF_COUNTRY_YT' => __('Mayotte', 'fpf-framework'),
			'FPF_COUNTRY_MX' => __('Mexico', 'fpf-framework'),
			'FPF_COUNTRY_FM' => __('Federated States of Micronesia', 'fpf-framework'),
			'FPF_COUNTRY_MD' => __('Moldova, Republic of', 'fpf-framework'),
			'FPF_COUNTRY_MC' => __('Monaco', 'fpf-framework'),
			'FPF_COUNTRY_MN' => __('Mongolia', 'fpf-framework'),
			'FPF_COUNTRY_ME' => __('Montenegro', 'fpf-framework'),
			'FPF_COUNTRY_MS' => __('Montserrat', 'fpf-framework'),
			'FPF_COUNTRY_MA' => __('Morocco', 'fpf-framework'),
			'FPF_COUNTRY_MZ' => __('Mozambique', 'fpf-framework'),
			'FPF_COUNTRY_MM' => __('Myanmar', 'fpf-framework'),
			'FPF_COUNTRY_NA' => __('Namibia', 'fpf-framework'),
			'FPF_COUNTRY_NR' => __('Nauru', 'fpf-framework'),
			'FPF_COUNTRY_NP' => __('Nepal', 'fpf-framework'),
			'FPF_COUNTRY_NL' => __('Netherlands', 'fpf-framework'),
			'FPF_COUNTRY_AN' => __('Netherlands Antilles', 'fpf-framework'),
			'FPF_COUNTRY_NC' => __('New Caledonia', 'fpf-framework'),
			'FPF_COUNTRY_NZ' => __('New Zealand', 'fpf-framework'),
			'FPF_COUNTRY_NI' => __('Nicaragua', 'fpf-framework'),
			'FPF_COUNTRY_NE' => __('Niger', 'fpf-framework'),
			'FPF_COUNTRY_NG' => __('Nigeria', 'fpf-framework'),
			'FPF_COUNTRY_NU' => __('Niue', 'fpf-framework'),
			'FPF_COUNTRY_NF' => __('Norfolk Island', 'fpf-framework'),
			'FPF_COUNTRY_MP' => __('Northern Mariana Islands', 'fpf-framework'),
			'FPF_COUNTRY_NO' => __('Norway', 'fpf-framework'),
			'FPF_COUNTRY_OM' => __('Oman', 'fpf-framework'),
			'FPF_COUNTRY_PK' => __('Pakistan', 'fpf-framework'),
			'FPF_COUNTRY_PW' => __('Palau', 'fpf-framework'),
			'FPF_COUNTRY_PS' => __('Palestinian Territory', 'fpf-framework'),
			'FPF_COUNTRY_PA' => __('Panama', 'fpf-framework'),
			'FPF_COUNTRY_PG' => __('Papua New Guinea', 'fpf-framework'),
			'FPF_COUNTRY_PY' => __('Paraguay', 'fpf-framework'),
			'FPF_COUNTRY_PE' => __('Peru', 'fpf-framework'),
			'FPF_COUNTRY_PH' => __('Philippines', 'fpf-framework'),
			'FPF_COUNTRY_PN' => __('Pitcairn', 'fpf-framework'),
			'FPF_COUNTRY_PL' => __('Poland', 'fpf-framework'),
			'FPF_COUNTRY_PT' => __('Portugal', 'fpf-framework'),
			'FPF_COUNTRY_PR' => __('Puerto Rico', 'fpf-framework'),
			'FPF_COUNTRY_QA' => __('Qatar', 'fpf-framework'),
			'FPF_COUNTRY_RE' => __('Reunion', 'fpf-framework'),
			'FPF_COUNTRY_RO' => __('Romania', 'fpf-framework'),
			'FPF_COUNTRY_RU' => __('Russian Federation', 'fpf-framework'),
			'FPF_COUNTRY_RW' => __('Rwanda', 'fpf-framework'),
			'FPF_COUNTRY_SH' => __('Saint Helena', 'fpf-framework'),
			'FPF_COUNTRY_KN' => __('Saint Kitts and Nevis', 'fpf-framework'),
			'FPF_COUNTRY_LC' => __('Saint Lucia', 'fpf-framework'),
			'FPF_COUNTRY_PM' => __('Saint Pierre and Miquelon', 'fpf-framework'),
			'FPF_COUNTRY_VC' => __('Saint Vincent and the Grenadines', 'fpf-framework'),
			'FPF_COUNTRY_WS' => __('Samoa', 'fpf-framework'),
			'FPF_COUNTRY_SM' => __('San Marino', 'fpf-framework'),
			'FPF_COUNTRY_ST' => __('Sao Tome and Principe', 'fpf-framework'),
			'FPF_COUNTRY_SA' => __('Saudi Arabia', 'fpf-framework'),
			'FPF_COUNTRY_SN' => __('Senegal', 'fpf-framework'),
			'FPF_COUNTRY_RS' => __('Serbia', 'fpf-framework'),
			'FPF_COUNTRY_SC' => __('Seychelles', 'fpf-framework'),
			'FPF_COUNTRY_SL' => __('Sierra Leone', 'fpf-framework'),
			'FPF_COUNTRY_SG' => __('Singapore', 'fpf-framework'),
			'FPF_COUNTRY_SK' => __('Slovakia', 'fpf-framework'),
			'FPF_COUNTRY_SI' => __('Slovenia', 'fpf-framework'),
			'FPF_COUNTRY_SB' => __('Solomon Islands', 'fpf-framework'),
			'FPF_COUNTRY_SO' => __('Somalia', 'fpf-framework'),
			'FPF_COUNTRY_ZA' => __('South Africa', 'fpf-framework'),
			'FPF_COUNTRY_GS' => __('South Georgia and the South Sandwich Islands', 'fpf-framework'),
			'FPF_COUNTRY_ES' => __('Spain', 'fpf-framework'),
			'FPF_COUNTRY_LK' => __('Sri Lanka', 'fpf-framework'),
			'FPF_COUNTRY_SD' => __('Sudan', 'fpf-framework'),
			'FPF_COUNTRY_SS' => __('South Sudan', 'fpf-framework'),
			'FPF_COUNTRY_SR' => __('Suriname', 'fpf-framework'),
			'FPF_COUNTRY_SJ' => __('Svalbard and Jan Mayen', 'fpf-framework'),
			'FPF_COUNTRY_SZ' => __('Swaziland', 'fpf-framework'),
			'FPF_COUNTRY_SE' => __('Sweden', 'fpf-framework'),
			'FPF_COUNTRY_CH' => __('Switzerland', 'fpf-framework'),
			'FPF_COUNTRY_SY' => __('Syrian Arab Republic', 'fpf-framework'),
			'FPF_COUNTRY_TW' => __('Taiwan', 'fpf-framework'),
			'FPF_COUNTRY_TJ' => __('Tajikistan', 'fpf-framework'),
			'FPF_COUNTRY_TZ' => __('Tanzania, United Republic of', 'fpf-framework'),
			'FPF_COUNTRY_TH' => __('Thailand', 'fpf-framework'),
			'FPF_COUNTRY_TL' => __('Timor-Leste', 'fpf-framework'),
			'FPF_COUNTRY_TG' => __('Togo', 'fpf-framework'),
			'FPF_COUNTRY_TK' => __('Tokelau', 'fpf-framework'),
			'FPF_COUNTRY_TO' => __('Tonga', 'fpf-framework'),
			'FPF_COUNTRY_TT' => __('Trinidad and Tobago', 'fpf-framework'),
			'FPF_COUNTRY_TN' => __('Tunisia', 'fpf-framework'),
			'FPF_COUNTRY_TR' => __('Turkey', 'fpf-framework'),
			'FPF_COUNTRY_TM' => __('Turkmenistan', 'fpf-framework'),
			'FPF_COUNTRY_TC' => __('Turks and Caicos Islands', 'fpf-framework'),
			'FPF_COUNTRY_TV' => __('Tuvalu', 'fpf-framework'),
			'FPF_COUNTRY_UG' => __('Uganda', 'fpf-framework'),
			'FPF_COUNTRY_UA' => __('Ukraine', 'fpf-framework'),
			'FPF_COUNTRY_AE' => __('United Arab Emirates', 'fpf-framework'),
			'FPF_COUNTRY_GB' => __('United Kingdom', 'fpf-framework'),
			'FPF_COUNTRY_US' => __('United States', 'fpf-framework'),
			'FPF_COUNTRY_UM' => __('United States Minor Outlying Islands', 'fpf-framework'),
			'FPF_COUNTRY_UY' => __('Uruguay', 'fpf-framework'),
			'FPF_COUNTRY_UZ' => __('Uzbekistan', 'fpf-framework'),
			'FPF_COUNTRY_VU' => __('Vanuatu', 'fpf-framework'),
			'FPF_COUNTRY_VE' => __('Venezuela', 'fpf-framework'),
			'FPF_COUNTRY_VN' => __('Vietnam', 'fpf-framework'),
			'FPF_COUNTRY_VG' => __('Virgin Islands, British', 'fpf-framework'),
			'FPF_COUNTRY_VI' => __('Virgin Islands, U.S.', 'fpf-framework'),
			'FPF_COUNTRY_WF' => __('Wallis and Futuna', 'fpf-framework'),
			'FPF_COUNTRY_EH' => __('Western Sahara', 'fpf-framework'),
			'FPF_COUNTRY_YE' => __('Yemen', 'fpf-framework'),
			'FPF_COUNTRY_ZM' => __('Zambia', 'fpf-framework'),
			'FPF_COUNTRY_ZW' => __('Zimbabwe', 'fpf-framework'),
			'FPF_SMART_TAGS_TITLE' => __('Smart Tags', 'fpf-framework'),
			'FPF_TAG_CLIENTDEVICE' => __('Visitor Device Type', 'fpf-framework'),
			'FPF_TAG_CLIENTOS' => __('Visitor Operating System', 'fpf-framework'),
			'FPF_TAG_CLIENTBROWSER' => __('Visitor Browser', 'fpf-framework'),
			'FPF_TAG_CLIENTUSERAGENT' => __('Visitor Agent String', 'fpf-framework'),
			'FPF_TAG_IP' => __('Visitor IP Address', 'fpf-framework'),
			'FPF_TAG_URL' => __('Page URL', 'fpf-framework'),
			'FPF_TAG_URLENCODED' => __('Page URL Encoded', 'fpf-framework'),
			'FPF_TAG_URLPATH' => __('Page Path', 'fpf-framework'),
			'FPF_TAG_REFERRER' => __('Page Referrer', 'fpf-framework'),
			'FPF_TAG_SITENAME' => __('Site Name', 'fpf-framework'),
			'FPF_TAG_SITEEMAIL' => __('Site Email', 'fpf-framework'),
			'FPF_TAG_SITEURL' => __('Site URL', 'fpf-framework'),
			'FPF_TAG_PAGETITLE' => __('Page Title', 'fpf-framework'),
			'FPF_TAG_PAGEDESC' => __('Page Meta Description', 'fpf-framework'),
			'FPF_TAG_PAGELANG' => __('Page Language Code', 'fpf-framework'),
			'FPF_TAG_PAGELANGURL' => __('Page Language Code URL', 'fpf-framework'),
			'FPF_TAG_PAGEBROWSERTITLE' => __('Browser Title', 'fpf-framework'),
			'FPF_TAG_USERID' => __('User ID', 'fpf-framework'),
			'FPF_TAG_USERNAME' => __('User Full Name', 'fpf-framework'),
			'FPF_TAG_USERLOGIN' => __('User Login', 'fpf-framework'),
			'FPF_TAG_USEREMAIL' => __('User Email', 'fpf-framework'),
			'FPF_TAG_USERFIRSTNAME' => __('User First Name', 'fpf-framework'),
			'FPF_TAG_USERLASTNAME' => __('User Last Name', 'fpf-framework'),
			'FPF_TAG_USERROLES' => __('User Roles', 'fpf-framework'),
			'FPF_TAG_USERREGISTERDATE' => __('User Register Date', 'fpf-framework'),
			'FPF_TAG_DATE' => __('Date', 'fpf-framework'),
			'FPF_TAG_TIME' => __('Time', 'fpf-framework'),
			'FPF_TAG_RANDOMID' => __('Random ID', 'fpf-framework'),
			'FPF_TAG_DAY' => __('Day', 'fpf-framework'),
			'FPF_TAG_MONTH' => __('Month', 'fpf-framework'),
			'FPF_TAG_YEAR' => __('Year', 'fpf-framework'),
			'FPF_TAG_EDDCARTCOUNT' => __('Easy Digital Downloads Cart Count', 'fpf-framework'),
			'FPF_TAG_EDDCARTTOTAL' => __('Easy Digital Downloads Cart Total', 'fpf-framework'),
			'FPF_TAG_EDDCARTSUBTOTAL' => __('Easy Digital Downloads Cart Subtotal', 'fpf-framework'),
			'FPF_TAG_EDDSTOCK' => __('Easy Digital Downloads Cart Stock', 'fpf-framework'),
			'FPF_TAG_WOOCARTCOUNT' => __('WooCommerce Cart Count', 'fpf-framework'),
			'FPF_TAG_WOOCARTTOTAL' => __('WooCommerce Cart Total', 'fpf-framework'),
			'FPF_TAG_WOOCARTSUBTOTAL' => __('WooCommerce Cart Subtotal', 'fpf-framework'),
			'FPF_TAG_WOOSTOCK' => __('WooCommerce Cart Stock', 'fpf-framework'),
			'FPF_CLOSE' => __('Close', 'fpf-framework'),
			'FPF_SYNTAX' => __('Syntax', 'fpf-framework'),
			'FPF_DESC' => __('Description', 'fpf-framework'),
			'FPF_OUTPUT_EXAMPLE' => __('Output example', 'fpf-framework'),
			'FPF_LANGUAGE' => __('Language', 'fpf-framework'),
			'FPF_WPML_LANGUAGE' => __('WPML Language', 'fpf-framework'),
			'FPF_LANGUAGE_DESC' => __('Target visitors who are browsing your website in a specific language', 'fpf-framework'),
			'FPF_TAG_FBID' => __('Campaign ID', 'fpf-framework'),
			'FPF_TAG_FBTITLE' => __('Campaign Title', 'fpf-framework'),
			'FPF_EXPORT' => __('Export', 'fpf-framework'),
			'FPF_GENERAL' => __('General', 'fpf-framework'),
			'FPF_AS_EXPORTED' => __('As Exported', 'fpf-framework'),
			'FPF_SELECT_IMPORT_FILE' => __('Select Import File', 'fpf-framework'),
			'FPF_DUPLICATE' => __('Duplicate', 'fpf-framework'),
			'FPF_PERCENTAGE' => __('Percentage', 'fpf-framework'),
			'FPF_PIXEL' => __('Pixel', 'fpf-framework'),
			'FPF_DOWN' => __('Down', 'fpf-framework'),
			'FPF_UP' => __('Up', 'fpf-framework'),
			'FPF_DESIGN' => __('Design', 'fpf-framework'),
			'FPF_ADD_ITEM' => __('Add Item', 'fpf-framework'),
			'FPF_GOAL' => __('Goal', 'fpf-framework'),
			'FPF_EVENT' => __('Event', 'fpf-framework'),
			'FPF_EVENTS' => __('Events', 'fpf-framework'),
			'FPF_SELECT_AN_EVENT' => __('Select an event', 'fpf-framework'),
			'FPF_BEFORE_OPEN' => __('Before Open', 'fpf-framework'),
			'FPF_OPEN' => __('Open', 'fpf-framework'),
			'FPF_TOGGLE' => __('Toggle', 'fpf-framework'),
			'FPF_AFTER_OPEN' => __('After Open', 'fpf-framework'),
			'FPF_BEFORE_CLOSE' => __('Before Close', 'fpf-framework'),
			'FPF_AFTER_CLOSE' => __('After Close', 'fpf-framework'),
			'FPF_ACTION' => __('Action', 'fpf-framework'),
			'FPF_ACTION_NAME' => __('Action Name', 'fpf-framework'),
			'FPF_PLEASE_SELECT_A_FILE_TO_UPLOAD' => __('Please select a file to upload.', 'fpf-framework'),
			'FPF_PLEASE_CHOOSE_A_VALID_FILE' => __('Please choose a valid file.', 'fpf-framework'),
			'FPF_FILE_EMPTY' => __('File is empty!', 'fpf-framework'),
			'FPF_ITEMS_SAVED' => __('Items saved!', 'fpf-framework'),
			'FPF_SETTINGS_SAVED' => __('Settings saved!', 'fpf-framework'),
			'FPF_MY_FAVORITES' => __('My Favorites', 'fpf-framework'),
			'FPF_OTHER' => __('Other', 'fpf-framework'),
			'FPF_PASSWORD' => __('Password', 'fpf-framework'),
			'FPF_REMEMBER_ME' => __('Remember Me', 'fpf-framework'),
			'FPF_LOG_IN' => __('Log in', 'fpf-framework'),
			'FPF_LOG_OUT' => __('Log out', 'fpf-framework'),
			'FPF_GUEST' => __('Guest', 'fpf-framework'),
			'FPF_PLEASE_SELECT' => __('Please Select', 'fpf-framework'),
			'FPF_GEOLOCATION_SERVICES' => __('Geolocation Services', 'fpf-framework'),
			'FPF_LICENSE' => __('License', 'fpf-framework'),
			'FPF_GEOIP_LICENSE_KEY_DESC' => __('Get your free License Key to download the latest MaxMind GeoLite2 Database.', 'fpf-framework'),
			'FPF_GEOLOCATION' => __('Geolocation', 'fpf-framework'),
			'FPF_GEOIP_GEOLOCATION_SERVICES' => __('Geolocation Services', 'fpf-framework'),
			'FPF_GEOIP_LICENSE_KEY_GET' => __('Get a free License Key', 'fpf-framework'),
			'FPF_GEOIP_GEOLOCATION_SERVICES_HEADING_DESC' => __('Geolocation services provide features such as (finding out the country of an IP address as well as retrieving the country data for Analytics) for Fireplugins.com plugins. Without it the geolocation features will not be available. This includes GeoLite2 data created by MaxMind <a href="http://www.maxmind.com">http://www.maxmind.com</a>.', 'fpf-framework'),
			'FPF_GEOIP_UPDATE_DB' => __('Update Database', 'fpf-framework'),
			'FPF_GEOIP_UPDATE_DB_DESC' => __('Update the GeoLite2 database from MaxMind servers. This might take several seconds to finish. Please be patient.', 'fpf-framework'),
			'FPF_GEOIP_DATABASE_UPDATED' => __('GeoIP database successfully updated!', 'fpf-framework'),
			'FPF_GEOIP_LICENSE_KEY_EMPTY' => __('Please enter a valid License Key.', 'fpf-framework'),
			'FPF_GEOIP_ERR_MAXMIND_GENERIC' => __('A connection error occurred. Please retry updating the GeoLite2 Country database in 24 hours.', 'fpf-framework'),
			'FPF_GEOIP_ERR_EMPTYDOWNLOAD' => __('Downloading the GeoLite2 Country database failed: empty file retrieved from server. Please contact your host.', 'fpf-framework'),
			'FPF_GEOIP_ERR_UNAUTHORIZED' => __('You have supplied an invalid MaxMind license key.', 'fpf-framework'),
			'FPF_GEOIP_ERR_CANTWRITE' => __('Insufficient rights while trying to save GeoIP database.', 'fpf-framework'),
			'FPF_GEOIP_ERR_INVALIDDB' => __('Downloaded database seems to be invalid. Please retry updating the GeoLite2 Country database in 24 hours.', 'fpf-framework'),
			'FPF_LAST_UPDATED' => __('Last Updated', 'fpf-framework'),
			'FPF_GEOIP_LAST_UPDATED_DESC' => __('Indicates the last datetime the database updated.', 'fpf-framework'),
			'FPF_GEOIP_LOOKUP' => __('Lookup IP Address', 'fpf-framework'),
			'FPF_GEOIP_LOOKUP_DESC' => __('Test drive the GeoIP plugin by looking up an IP address.', 'fpf-framework'),
			'FPF_LOOKUP' => __('Lookup', 'fpf-framework'),
			'FPF_LICENSE_KEY' => __('License Key', 'fpf-framework'),
			'FPF_LICENSE_KEY_DESC' => __('To find your License Key log-in to <a href="https://www.fireplugins.com" target="_blank">FirePlugins</a> and then go to your downloads section.', 'fpf-framework'),
			'FPF_FIND_LICENSE_KEY' => __('Find License Key', 'fpf-framework'),
			'FPF_DATA' => __('Data', 'fpf-framework'),
			'FPF_GEOIP_MAINTENANCE' => __('GeoIP Database Maintenance', 'fpf-framework'),
			'FPF_GEOIP_MAINTENANCE_DESC' => __('%s finds the country of your visitors\' IP addresses using the MaxMind GeoLite2 Country database. You are advised to update it at least once per month. On most servers you can perform the update by clicking the button below.', 'fpf-framework'),
			'FPF_GEOIP_MAINTENANCE_WITHOUT_BTN_MENTION_DESC' => __('%s finds the country of your visitors\' IP addresses using the MaxMind GeoLite2 Country database. You are advised to update it at least once per month.', 'fpf-framework'),
			'FPF_UPGRADE_TO_PRO' => __('Upgrade to Pro', 'fpf-framework'),
			'FPF_UPGRADE' => __('Upgrade', 'fpf-framework'),
			'FPF_VIEW_PLANS' => __('View Plans', 'fpf-framework'),
			'FPF_UNLOCK' => __('Unlock', 'fpf-framework'),
			'FPF_FREE' => __('Free', 'fpf-framework'),
			'FPF_PRO' => __('Pro', 'fpf-framework'),
			'FPF_PRO_MODAL_IS_PRO_FEATURE' => __('<em class="pro-feature-name"></em> is a PRO Feature', 'fpf-framework'),
			'FPF_PRO_MODAL_WERE_SORRY' => __('We\'re sorry, <em class="pro-feature-name"></em> is not available on your plan. Please upgrade to the PRO plan to unlock all these awesome features.', 'fpf-framework'),
			'FPF_PRO_MODAL_UPGRADE_TO_PRO_VERSION' => __('Awesome! Only one step left. Click on the button below to complete the upgrade to the Pro version.', 'fpf-framework'),
			'FPF_PRO_MODAL_PERCENTAGE_OFF' => __('<strong>Bonus</strong>: %s lite users get <strong class="percentage">20%%</strong> off regular price, automatically applied at checkout.', 'fpf-framework'),
			'FPF_PRO_MODAL_PRESALES_QUESTIONS' => __('Pre-Sales questions? <a target="_blank" href="%s">Ask here</a>', 'fpf-framework'),
			'FPF_PRO_MODAL_UNLOCK_PRO_FEATURES' => __('Already purchased Pro? Learn how to <a target="_blank" href="%s">Unlock Pro Features</a>', 'fpf-framework'),
			'FPF_CLEAR' => __('Clear', 'fpf-framework'),
			'FPF_SAVE' => __('Save', 'fpf-framework'),
			'FPF_TOP' => __('Top', 'fpf-framework'),
			'FPF_RIGHT' => __('Right', 'fpf-framework'),
			'FPF_BOTTOM' => __('Bottom', 'fpf-framework'),
			'FPF_LEFT' => __('Left', 'fpf-framework'),
			'FPF_TOP_LEFT' => __('Top Left', 'fpf-framework'),
			'FPF_TOP_RIGHT' => __('Top Right', 'fpf-framework'),
			'FPF_BOTTOM_RIGHT' => __('Bottom Right', 'fpf-framework'),
			'FPF_BOTTOM_LEFT' => __('Bottom Left', 'fpf-framework'),
			'FPF_TAG_QUERYSTRINGPOST' => __('Post', 'fpf-framework'),
			'FPF_TAG_QUERYSTRINGACTION' => __('Action', 'fpf-framework'),
			'FPF_TAG_QUERYSTRINGLANG' => __('Lang', 'fpf-framework'),
			'FPF_ENTER_VALID_IP_ADDRESS' => __('Please enter a valid IP address', 'fpf-framework'),
			'FPF_LOADING' => __('Loading...', 'fpf-framework'),
			'FPF_LOADING_ANALYTICS' => __('Loading Analytics...', 'fpf-framework'),
			'FPF_INVALID_IP_ADDRESS' => __('Invalid IP Address', 'fpf-framework'),
			'FPF_DOWNLOADING_UPDATES_PLEASE_WAIT' => __('Downloading Updates. Please wait...', 'fpf-framework'),
			'FPF_DATABASE_UPDATED' => __('Database updated!', 'fpf-framework'),
			'FPF_BUTTON_LABEL' => __('Button Label', 'fpf-framework'),
			'FPF_BUTTON_CLASSES' => __('Button Classes', 'fpf-framework'),
			'FPF_PREVENT_DEFAULT_ACTION' => __('Prevent Default Action', 'fpf-framework'),
			'FPF_ANALYTICS' => __('Analytics', 'fpf-framework'),
			'FPF_DATE' => __('Date', 'fpf-framework'),
			'FPF_DATES' => __('Dates', 'fpf-framework'),
			'FPF_FILTER' => __('Filter', 'fpf-framework'),
			'FPF_COMPARE' => __('Compare', 'fpf-framework'),
			'FPF_CUSTOM' => __('Custom', 'fpf-framework'),
			'FPF_DATE_RANGE' => __('Date Range', 'fpf-framework'),
			'FPF_CANCEL' => __('Cancel', 'fpf-framework'),
			'FPF_APPLY' => __('Apply', 'fpf-framework'),
			'FPF_LEARN_MORE' => __('Learn more', 'fpf-framework'),
			'FPF_START_DATE' => __('Start Date', 'fpf-framework'),
			'FPF_END_DATE' => __('End Date', 'fpf-framework'),
			'FPF_URLS_CONTAINING' => __('URLs containing', 'fpf-framework'),
			'FPF_URLS_NOT_CONTAINING' => __('URLs not containing', 'fpf-framework'),
			'FPF_EXACT_URL' => __('Exact URL', 'fpf-framework'),
			'FPF_EXAMPLE_LINK_HINT' => __('https://www.example.com/', 'fpf-framework'),
			'FPF_SELECT_A_FILTER_TYPE' => __('Select a filter type', 'fpf-framework'),
			'FPF_SELECT_A_DEVICE' => __('Select a device', 'fpf-framework'),
			'FPF_SELECT_A_COUNTRY' => __('Start searching for a country', 'fpf-framework'),
			'FPF_VS_LC' => __('vs', 'fpf-framework'),
			'FPF_NO_DATA' => __('No data', 'fpf-framework'),
			'FPF_IMPRESSIONS_PREVIOUS_PERIOD' => __('Impressions Previous Period', 'fpf-framework'),
			'FPF_GO_PRO' => __('Go Pro', 'fpf-framework'),
			'FPF_UNLOCK_MORE_FEATURES_WITH_PRO' => __('Unlock more features by going Pro!', 'fpf-framework'),
			'FPF_UNLOCK_MORE_FEATURES_WITH_PRO_READ_MORE' => __('Unlock more features by going Pro! Click to upgrade to the Pro version and increase your conversions!', 'fpf-framework'),
			'FPF_DAY' => __('Day', 'fpf-framework'),
			'FPF_WEEK' => __('Week', 'fpf-framework'),
			'FPF_MONTH' => __('Month', 'fpf-framework'),
			'FPF_DIFFERENCE' => __('Difference', 'fpf-framework'),
			'FPF_TABLE_DATA_OMITTED' => __('Table data omitted as there are no additional information to display.', 'fpf-framework'),
			'FPF_AVG_TIME_OPEN' => __('Average Time Open', 'fpf-framework'),
			'FPF_VIEW_MORE' => __('View More', 'fpf-framework'),
			'FPF_ADVANCED_ANALYTICS' => __('Advanced Analytics', 'fpf-framework'),
			'FPF_PLUGIN_MISSING' => __('Plugin <strong>%s</strong> is missing.', 'fpf-framework'),
			'FPF_PLUGIN_MISSING_INSTALL_TO_USE_FEATURE' => __('Plugin <strong>%s</strong> is missing. Please install it to use %s.', 'fpf-framework'),
			'FPF_ANALYZE' => __('Analyze', 'fpf-framework'),
			'FPF_PERCENTAGE_CHANGE_OVER_PREVIOUS_PERIOD' => __('Percentage change over previous period', 'fpf-framework'),
			'FPF_ANALYTICS_TABLE_NO_DATA_TO_DISPLAY' => __('There are no items to display.', 'fpf-framework'),
			'FPF_SELECT_A_PERIOD' => __('Select a period', 'fpf-framework'),
			'FPF_ANALYTICS_OVERVIEW_LOADING' => __('Loading Analytics Overview...', 'fpf-framework'),
			'FPF_REFERRER' => __('Referrer', 'fpf-framework'),
			'FPF_REFERRERS' => __('Referrers', 'fpf-framework'),
			'FPF_6_MONTHS' => __('6 Months', 'fpf-framework'),
			'FPF_1_YEAR' => __('1 Year', 'fpf-framework'),
			'FPF_2_YEARS' => __('2 Years', 'fpf-framework'),
			'FPF_5_YEARS' => __('5 Years', 'fpf-framework'),
			'FPF_KEEP_FOREVER' => __('Keep forever', 'fpf-framework'),
			'FPF_DATE_PERIOD' => __('Date Period', 'fpf-framework'),
			'FPF_FEATURE_IMAGE_UPGRADE_PRO_MSG1' => __('<strong>%s</strong> is a Pro feature!', 'fpf-framework'),
			'FPF_FEATURE_IMAGE_UPGRADE_PRO_MSG2' => __('Upgrade to Pro to unlock this feature.', 'fpf-framework'),
			'FPF_%S' => __('%s', 'fpf-framework'),
			'FPF_ADD_NEW' => __('Add New', 'fpf-framework'),
			'FPF_ADD_NEW_%S' => __('Add New %s', 'fpf-framework'),
			'FPF_EDIT_%S' => __('Edit %s', 'fpf-framework'),
			'FPF_NEW_%S' => __('New %s', 'fpf-framework'),
			'FPF_VIEW_%S' => __('View %s', 'fpf-framework'),
			'FPF_SEARCH' => __('Search', 'fpf-framework'),
			'FPF_SEARCH_%S' => __('Search %s', 'fpf-framework'),
			'FPF_NO_%S_FOUND' => __('No %s found', 'fpf-framework'),
			'FPF_NO_%S_FOUND_IN_TRASH' => __('No %s found in Trash', 'fpf-framework'),
			'FPF_PARENT_%S' => __('Parent %s:', 'fpf-framework'),
			'FPF_RESET' => __('Reset', 'fpf-framework'),
			'FPF_RESET_COLOR' => __('Reset color', 'fpf-framework'),
			'FPF_DEFAULT' => __('Default', 'fpf-framework'),
			'FPF_SELECT_COLOR' => __('Select color', 'fpf-framework'),
			'FPF_SELECT_DEFAULT_COLOR' => __('Select default color', 'fpf-framework'),
			'FPF_COLOR_VALUE' => __('Color value', 'fpf-framework'),
			'FPF_PHP_VERSION_FAIL' => __('%s requires PHP version %s+, please upgrade to the mentioned PHP version in order for the plugin to work.', 'fpf-framework'),
			'FPF_TEMPLATE' => __('Template', 'fpf-framework'),
			'FPF_KEEP_DATA_ON_UNINSTALL' => __('Keep data on uninstall', 'fpf-framework'),
			'FPF_KEEP_DATA_ON_UNINSTALL_DESC' => __('<strong>Enable</strong> to preserve the data when you uninstall the plugin.<br><strong>Disable</strong> to remove the data when you uninstall the plugin.', 'fpf-framework'),
			'FPF_HI' => __('Hi', 'fpf-framework'),
			'FPF_MADE_WITH_LOVE_BY_FP' => __('Made with ♥️ by FirePlugins', 'fpf-framework'),
			'FPF_GET_PRO_FEATURES' => __('Get PRO Features', 'fpf-framework'),
			'FPF_ROADMAP' => __('Roadmap', 'fpf-framework'),
			'FPF_TERMS_OF_USE' => __('Terms of Use', 'fpf-framework'),
			'FPF_PRIVACY_POLICY' => __('Privacy Policy', 'fpf-framework'),
			'FPF_WRITE_REVIEW' => __('Leave a review on WordPress Plugin Directory', 'fpf-framework'),
			'FPF_LIKE_PLUGIN' => __('Like %s?', 'fpf-framework'),
			'FPF_GO_TO_FP_SITE' => __('Go to FirePlugins Site', 'fpf-framework'),
			'FPF_MEDIA' => __('Media', 'fpf-framework'),
			'FPF_UNINSTALL' => __('Uninstall', 'fpf-framework'),
			'FPF_SETTINGS_UNINSTALL_DESC' => __('Select whether to keep or delete your settings on plugin uninstall.', 'fpf-framework'),
			'FPF_SETTINGS_LICENSE_DESC' => __('Activate your license to get automatic updates.', 'fpf-framework'),
			'FPF_DIMENSIONS_FIELD_LINK_VALUES_TITLE' => __('Enable to automatically fill in all values.', 'fpf-framework'),
			'FPF_REVIEW_REMINDER_MSG1' => __('%1$s Thank you for using our plugin, %2$s%3$s%4$s! %5$s You rock! Could you please do us a BIG favor and give it a 5-star rating on WordPress? This will help us spread the word and boost our motivation. %6$s - %7$s FirePlugins %8$s %9$s', 'fpf-framework'),
            'FPF_REVIEW_REMINDER_MSG2' => __('%1$s %2$s Yes, take me there! %3$s %4$s Remind me later %3$s %5$s I\'ve already done this  %3$s %6$s', 'fpf-framework'),
			'FPF_UNITS_TITLE' => __('Select the unit for this field.', 'fpf-framework'),
			'FPF_FORGOT_YOUR_PASSWORD' => __('Forgot your password?', 'fpf-framework'),
			'FPF_REFERRER_POPULAR_LIST_SELECTION' => __('Popular referrers', 'fpf-framework'),
			'FPF_REFERRER_POPULAR_LIST_SELECTION_DESC' => __('Select a referrer from a pre-defined list of popular referrers', 'fpf-framework'),
			'FPF_RENDER' => __('Render', 'fpf-framework'),
			'FPF_STATE' => __('State', 'fpf-framework'),
			'FPF_STATUS' => __('Status', 'fpf-framework'),
			'FPF_MISC' => __('Misc', 'fpf-framework'),
			'FPF_MISC_SETTINGS_DESC' => __('Configure miscellaneous settings in this section.', 'fpf-framework'),
			'FPF_NEVER' => __('Never', 'fpf-framework'),
			'FPF_SECONDS' => __('Seconds', 'fpf-framework'),
			'FPF_SECONDS_SHORT' => __('Secs', 'fpf-framework'),
			'FPF_MINUTES_SHORT' => __('Mins', 'fpf-framework'),
			'FPF_MINUTES' => __('Minutes', 'fpf-framework'),
			'FPF_HOURS' => __('Hours', 'fpf-framework'),
			'FPF_HOURS_SHORT' => __('Hrs', 'fpf-framework'),
			'FPF_DAYS' => __('Days', 'fpf-framework'),
			'FPF_ALWAYS_SHOW' => __('Always Show', 'fpf-framework'),
			'FPF_START_FROM_SCRATCH' => __('Start from scratch', 'fpf-framework'),
			'FPF_PREVIEW_TEMPLATE' => __('Preview template', 'fpf-framework'),
			'FPF_LIBRARY_SAVE_TEMPLATE_FAVORITES' => __('Save template to favorites', 'fpf-framework'),
			'FPF_LOADING_TEMPLATES' => __('Loading Templates...', 'fpf-framework'),
			'FPF_REFRESH_TEMPLATES' => __('Refresh Templates', 'fpf-framework'),
			'FPF_NO_RESULTS_FOUND' => __('No Results Found', 'fpf-framework'),
			'FPF_OOPS_NO_MATCHES_FOUND' => __('<strong>Ooops!</strong> No matches found.', 'fpf-framework'),
			'FPF_CUSTOM_SOUND_FILE' => __('Custom Sound File', 'fpf-framework'),
			'FPF_CUSTOM_SOUND_URL' => __('Custom Sound URL', 'fpf-framework'),
			'FPF_ONCE' => __('Once', 'fpf-framework'),
			'FPF_UNLIMITED' => __('Unlimited', 'fpf-framework'),
			'FPF_FREQUENCY' => __('Frequency', 'fpf-framework'),
			'FPF_SHOW' => __('Show', 'fpf-framework'),
			'FPF_DONT_SHOW_AGAIN_FOR' => __('Don\'t show again for', 'fpf-framework'),
			'FPF_TIMES_PER' => __('Times Per', 'fpf-framework'),
			'FPF_SHOW_FREQUENCY' => __('Show Frequency', 'fpf-framework'),
			'FPF_EVERY_TIME_PAGE_LOADS' => __('Every Time Page Loads', 'fpf-framework'),
			'FPF_GEO_DB_HAS_NOT_BEEN_UPDATED' => __('Database has not been updated.', 'fpf-framework'),
			'FPF_BEHAVIOR' => __('Behavior', 'fpf-framework'),
			'FPF_ALIGN_CONTENT' => __('Align Content', 'fpf-framework'),
			'FPF_NEW_ACTION_LABEL' => __('New Action Label', 'fpf-framework'),
			'FPF_EDIT_ITEM' => __('Edit item', 'fpf-framework'),
			'FPF_REMOVE_ITEM' => __('Remove item', 'fpf-framework'),
			'FPF_MOVE_ITEM' => __('Reorder item', 'fpf-framework'),
			'FPF_ACTIVATE_LICENSE' => __('Activate License', 'fpf-framework'),
			'FPF_DEACTIVATE_LICENSE' => __('Deactivate License', 'fpf-framework'),
			'FPF_X_VERSION_IS_AVAILABLE' => __('%s is now available', 'fpf-framework'),
			'FPF_AN_UPDATED_VERSION_IS_AVAILABLE' => __('A new and improved version of %s released on %s. Update Now!', 'fpf-framework'),
			'FPF_UPDATE' => __('Update', 'fpf-framework'),
			'FPF_UPDATE_NOW' => __('Update now', 'fpf-framework'),
			'FPF_PRO_TEMPLATES' => __('Pro Templates', 'fpf-framework'),
			'FPF_TEMPLATES_CANNOT_BE_RETRIEVED' => __('Cannot retrieve templates. Please try again.', 'fpf-framework'),
			'FPF_HIDE_PRO_TEMPLATES' => __('Hide Pro Templates', 'fpf-framework'),
			'FPF_ENJOY' => __('Enjoy', 'fpf-framework'),
			'FPF_NO_LICENSE_NEEDED' => __('No license needed', 'fpf-framework'),
			'FPF_VALID' => __('valid', 'fpf-framework'),
			'FPF_INVALID' => __('invalid', 'fpf-framework'),
			'FPF_YOUR_LICENSE_KEY_IS_STATUS' => __('Your license key is <strong class="fpf-%s-color">%s</strong>.', 'fpf-framework'),
			'FPF_PLEASE_ENTER_A_VALID_LICENSE_KEY_TO_RECEIVE_UPDATES' => __('Please enter a valid license key to receive updates.', 'fpf-framework'),
			'FPF_ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_ITEM' => __('Are you sure you want to delete this item?', 'fpf-framework'),
			'FPF_STAR' => __('Star', 'fpf-framework'),
			'FPF_CB_SELECT_CONDITION' => 'Select Condition', 'fpf-framework',
			'FPF_CB_NEW_CONDITION_GROUP' => 'New Condition Set', 'fpf-framework',
			'FPF_CB_SELECT_CONDITION_GET_STARTED' => 'Select a condition to get started.', 'fpf-framework',
			'FPF_CB_TRASH_CONDITION' => 'Trash Condition', 'fpf-framework',
			'FPF_CB_TRASH_CONDITION_GROUP' => 'Trash Condition Group', 'fpf-framework',
			'FPF_CB_ADD_CONDITION' => 'Add Condition', 'fpf-framework',
			'FPF_CB_SHOW_WHEN' => 'Display when', 'fpf-framework',
			'FPF_CB_OF_THE_CONDITIONS_MATCH' => 'of the conditions below are met', 'fpf-framework',
			'FPF_PHP_COLLECTION_SCRIPTS' => 'A collection of ready-to-use PHP assignment scripts is available. <a target="_blank" href="http://www.fireplugins.com/docs/general/extending-general/custom-php-assignments">View collection</a>', 'fpf-framework',
			'FPF_IS' => 'Is', 'fpf-framework',
			'FPF_IS_NOT' => 'Is not', 'fpf-framework',
			'FPF_IS_EMPTY' => 'Is empty', 'fpf-framework',
			'FPF_IS_NOT_EMPTY' => 'Is not empty', 'fpf-framework',
			'FPF_IS_BETWEEN' => 'Is between', 'fpf-framework',
			'FPF_IS_NOT_BETWEEN' => 'Is not between', 'fpf-framework',
			'FPF_DISPLAY_CONDITIONS_LOADING' => 'Loading Display Conditions...', 'fpf-framework',
			'FPF_CB_TOGGLE_RULE_GROUP_STATUS' => 'Enable or disable Condition Set', 'fpf-framework',
			'FPF_CB_TOGGLE_RULE_STATUS' => 'Enable or disable Condition', 'fpf-framework',
			'FPF_DISPLAY_CONDITIONS_HINT_DATE' => __('Your server\'s date time is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_TIME' => __('Your server\'s time is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_DAY' => __('Today is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_MONTH' => __('The current month is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_USERID' => __('The user you\'re logged-in is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_USERGROUP' => __('The User Groups assigned to the account you\'re logged-in are: %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_ACCESSLEVEL' => __('The Viewing Access Levels assigned to the account you\'re logged-in are: %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_DEVICE' => __('The type of the device you\'re using is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_BROWSER' => __('The browser you\'re using is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_OS' => __('The operating system you\'re using is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_GEO' => __('Based on your IP address (%s), the %s you\'re physically located in, is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_IP' => __('Your IP Address is %s.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_GEO_ERROR' => __('Based on your IP address (%s), we couldn\'t determine where you\'re physically located in.', 'fpf-framework'),
			'FPF_DISPLAY_CONDITIONS_HINT_REFERRER' => '',
			'FPF_DISPLAY_CONDITIONS_HINT_PHP' => '',
			'FPF_AND' => __('and', 'fpf-framework'),
			'FPF_LOADING_FILTERS' => __('Loading filters...', 'fpf-framework'),
			'FPF_FILTERS' => __('Filters', 'fpf-framework'),
			'FPF_CLEAR_ALL' => __('Clear all', 'fpf-framework'),
			'FPF_OPEN_SIDEBAR' => __('Open sidebar', 'fpf-framework'),
			'FPF_CLOSE_SIDEBAR' => __('Close sidebar', 'fpf-framework'),
			'FPF_SORT_BY' => __('Sort by', 'fpf-framework'),
			'FPF_POPULAR' => __('Popular', 'fpf-framework'),
			'FPF_TRENDING' => __('Trending', 'fpf-framework'),
			'FPF_SOLUTION' => __('Solution', 'fpf-framework'),
			'FPF_SOLUTIONS' => __('Solutions', 'fpf-framework'),
			'FPF_INSERT' => __('Insert', 'fpf-framework'),
			'FPF_INSERT_TEMPLATE_NOW' => __('Insert template now', 'fpf-framework'),
			'FPF_COMPATIBILITY' => __('Compatibility', 'fpf-framework'),
			'FPF_REQUIREMENTS' => __('Requirements', 'fpf-framework'),
			'FPF_SHOWING_RESULTS_FOR' => __('Showing <span class="fpf-showing-results-counter"></span> results for', 'fpf-framework'),
			'FPF_REQUEST_TEMPLATE' => __('Request template', 'fpf-framework'),
			'FPF_TEMPLATE_INFORMATION' => __('Click to view information about this template', 'fpf-framework'),
			'FPF_UPDATE_WORDPRESS' => __('Update WordPress', 'fpf-framework'),
			'FPF_UPDATE_WORDPRESS_TO_INSERT_TEMPLATE' => __('Update WordPress to insert this template', 'fpf-framework'),
			'FPF_UPDATE_PLUGIN' => __('Update Plugin', 'fpf-framework'),
			'FPF_UPDATE_PLUGIN_X' => __('Update %s', 'fpf-framework'),
			'FPF_UPDATE_PLUGIN_X_TO_INSERT_TEMPLATE' => __('Update %s to insert this template', 'fpf-framework'),
			'FPF_INSTALL_PLUGIN' => __('Install Plugin', 'fpf-framework'),
			'FPF_INSTALL_PLUGIN_X' => __('Install %s', 'fpf-framework'),
			'FPF_INSTALL_PLUGIN_X_TO_INSERT_TEMPLATE' => __('Install %s to insert this template', 'fpf-framework'),
			'FPF_ACTIVATE_PLUGIN' => __('Activate Plugin', 'fpf-framework'),
			'FPF_ACTIVATE_PLUGIN_X' => __('Activate %s', 'fpf-framework'),
			'FPF_ACTIVATE_PLUGIN_X_TO_INSERT_TEMPLATE' => __('Activate %s to insert this template', 'fpf-framework'),
			'FPF_MULTIPLE_ISSUES_DETECTED' => __('Multiple issues detected', 'fpf-framework'),
			'FPF_NO_LICENSE_KEY_DETECTED' => __('No license key detected', 'fpf-framework'),
			'FPF_UPGRADE_TO_UC_PRO' => __('Upgrade to PRO', 'fpf-framework'),
			'FPF_CATEGORY' => __('Category', 'fpf-framework'),
			'FPF_THIRD_PARTY' => __('3rd Party', 'fpf-framework'),
			'FPF_WP' => __('WP', 'fpf-framework'),
			'FPF_WORDPRESS' => __('WordPress', 'fpf-framework'),
			'FPF_REQUIRED_ACTIONS' => __('Required Actions', 'fpf-framework'),
			'FPF_FEATURED' => __('Featured', 'fpf-framework'),
			'FPF_NEWEST' => __('Newest', 'fpf-framework'),
			'FPF_INVALID_LICENSE_KEY_ENTERED' => __('License key is invalid/expired.', 'fpf-framework'),
			'FPF_UPGRADE_TO_PRO_TO_UNLOCK_TEMPLATE' => __('Upgrade to PRO unlock this template', 'fpf-framework'),
			'FPF_INSERT_TEMPLATE' => __('Insert template', 'fpf-framework'),
			'FPF_DETECTED' => __('Detected', 'fpf-framework'),
			'FPF_CHECK' => __('Check', 'fpf-framework'),
			'FPF_LITE' => __('Lite', 'fpf-framework'),
			'FPF_SET_LICENSE_KEY' => __('Set License Key', 'fpf-framework'),
			'FPF_INVALID_EXPIRED_LICENSE_KEY' => __('Invalid/Expired License Key', 'fpf-framework'),
			'FPF_ERROR_PRO_TO_FREE' => __('You currently have the PRO version installed.', 'fpf-framework'),
			'FPF_ERROR_UNINSTALL_FIRST' => __('By upgrading to the latest FREE version, you will lose functionality.<br />If you want to keep all functionality you will have to <a href="https://www.fireplugins.com/%s" target="_blank">buy a PRO version</a>.<br /><br />To be able to upgrade to the new FREE version, you will have to uninstall <strong>%s</strong> first.', 'fpf-framework'),
			'FPF_ERROR_OCCURRED_PLEASE_TRY_AGAIN' => __('An error occurred, please try again.', 'fpf-framework'),
			'FPF_INVALID_LICENSE' => __('Invalid license.', 'fpf-framework'),
			'FPF_LICENSE_KEY_EXPIRED_ON' => __('Your license key expired on %s.', 'fpf-framework'),
			'FPF_LICENSE_KEY_REVOKED' => __('Your license key has been disabled.', 'fpf-framework'),
			'FPF_LICENSE_KEY_NOT_VALID_FOR_THIS_URL' => __('Your license is not active for this URL.', 'fpf-framework'),
			'FPF_LICENSE_MISMATCH' => __('This appears to be an invalid license key for %s.', 'fpf-framework'),
			'FPF_LICENSE_LIMIT_REACHED' => __('Your license key has reached its activation limit.', 'fpf-framework'),
			'FPF_ERROR_USER_ALREADY_EXIST' => __('%s is already a member.', 'fpf-framework'),
			'FPF_ERROR_INVALID_EMAIL_ADDRESS' => __('%s looks fake or invalid, please enter a real email address.', 'fpf-framework'),
			'FPF_SUBMISSIONS' => __('Submissions', 'fpf-framework'),
			'FPF_NAME_RAW' => __('Name', 'fpf-framework'),
			'FPF_ACTIONS' => __('Actions', 'fpf-framework'),
			'FPF_PUBLISH' => __('Publish', 'fpf-framework'),
			'FPF_UNPUBLISH' => __('Unpublish', 'fpf-framework'),
			'FPF_UNPUBLISHED' => __('Unpublished', 'fpf-framework'),
			'FPF_TRASH' => __('Trash', 'fpf-framework'),
			'FPF_VISITOR_ID' => __('Visitor ID', 'fpf-framework'),
			'FPF_EMAIL' => __('Email', 'fpf-framework'),
			'FPF_USER_NAME' => __('User Name', 'fpf-framework'),
			'FPF_USER' => __('User', 'fpf-framework'),
			'FPF_PUBLISHED' => __('Published', 'fpf-framework'),
			'FPF_CREATED' => __('Created', 'fpf-framework'),
			'FPF_TRASHED' => __('Trashed', 'fpf-framework'),
			'FPF_CANNOT_VALIDATE_REQUEST' => __('Cannot validate request.', 'fpf-framework'),
			'FPF_DELETE' => __('Delete', 'fpf-framework'),
			'FPF_BACK' => __('Back', 'fpf-framework'),
			'FPF_SELECT_A_LIST' => __('Select a list', 'fpf-framework'),
			'FPF_API_KEY_INVALID_OR_INTEGRATION_ACCOUNT_HAS_NO_LISTS' => __('Either your API Key is invalid or your %s account does not have any lists created.', 'fpf-framework'),
			'FPF_INTEGRATION_ACCOUNT_HAS_NO_LISTS' => __('Your %s account does not have any lists created.', 'fpf-framework'),
			'FPF_PLEASE_ENTER_AN_API_KEY' => __('Please enter an API Key.', 'fpf-framework'),
			'FPF_NO_INTEGRATION_SUPPLIED' => __('No Integration was supplied.', 'fpf-framework'),
			'FPF_CANNOT_VERIFY_REQUEST' => __('Cannot verify request.', 'fpf-framework'),
			'FPF_NO_SUCH_INTEGRATION_EXISTS' => __('No such integration exists.', 'fpf-framework'),
			'FPF_INTEGRATION_INVALID' => __('Integration is invalid.', 'fpf-framework'),
			'FPF_WHERE_TO_FIND_API_KEY' => __('Where can I find my API Key?', 'fpf-framework'),
			'FPF_PA_CITY_AS_DESC' => __('Enter a city name in English. You can enter multiple cities by separating them via a comma and/or \'enter\'.', 'fpf-framework'),
			'FPF_CONVERSION_RATE' => __('Conversion Rate', 'fpf-framework'),
			'FPF_CONVERSIONRATE' => __('Conversion Rate', 'fpf-framework'),
			'FPF_SUBMISSIONS_CHART' => __('Submissions Chart', 'fpf-framework'),
			'FPF_CONVERSION_RATE_CHART' => __('Conversion Rate Chart', 'fpf-framework'),
			'FPF_DAY_OF_WEEK' => __('Day of Week', 'fpf-framework'),
			'FPF_WEEKDAY' => __('Weekday', 'fpf-framework'),
			'FPF_TIME' => __('Time', 'fpf-framework'),
			'FPF_SELECT_CONDITION' => __('Select Condition', 'fpf-framework'),
			'FPF_REGION' => __('Region', 'fpf-framework'),
			'FPF_BROWSERS' => __('Browsers', 'fpf-framework'),
			'FPF_INTEGRATIONS' => __('Integrations', 'fpf-framework'),
			'FPF_DATETIME' => __('Date/Time', 'fpf-framework'),
			'FPF_TECHNOLOGY' => __('Technology', 'fpf-framework'),
			'FPF_TYPE_A_CONDITION' => __('Type a condition...', 'fpf-framework'),
			'FPF_MONDAY' => __('Monday', 'fpf-framework'),
			'FPF_TUESDAY' => __('Tuesday', 'fpf-framework'),
			'FPF_WEDNESDAY' => __('Wednesday', 'fpf-framework'),
			'FPF_THURSDAY' => __('Thursday', 'fpf-framework'),
			'FPF_FRIDAY' => __('Friday', 'fpf-framework'),
			'FPF_SATURDAY' => __('Saturday', 'fpf-framework'),
			'FPF_SUNDAY' => __('Sunday', 'fpf-framework'),
			'FPF_WEEKEND' => __('Weekend', 'fpf-framework'),
			'FPF_WEEKDAYS' => __('Weekdays', 'fpf-framework'),
			'FPF_JANUARY' => __('January', 'fpf-framework'),
			'FPF_FEBRUARY' => __('February', 'fpf-framework'),
			'FPF_MARCH' => __('March', 'fpf-framework'),
			'FPF_APRIL' => __('April', 'fpf-framework'),
			'FPF_MAY' => __('May', 'fpf-framework'),
			'FPF_JUNE' => __('June', 'fpf-framework'),
			'FPF_JULY' => __('July', 'fpf-framework'),
			'FPF_AUGUST' => __('August', 'fpf-framework'),
			'FPF_SEPTEMBER' => __('September', 'fpf-framework'),
			'FPF_OCTOBER' => __('October', 'fpf-framework'),
			'FPF_NOVEMBER' => __('November', 'fpf-framework'),
			'FPF_DECEMBER' => __('December', 'fpf-framework'),
			'FPF_TYPE_A_USER' => __('Start searching for a user', 'fpf-framework'),
			'FPF_ASSIGN_DEVICES_NOTE' => __('Keep in mind that device detection is not always 100% accurate. Users can setup their browser to mimic other devices', 'fpf-framework'),
			'FPF_SELECT_A_CONTINENT' => __('Start searching for a continent', 'fpf-framework'),
			'FPF_SELECT_A_FIREBOX_HINT' => __('Start searching for a FireBox campaign', 'fpf-framework'),
			'FPF_FIREBOX_VIEWED_ANOTHER_CAMPAIGN' => __('FireBox - Viewed Another Campaign', 'fpf-framework'),
			'FPF_REFERRER_NOTE_DESC' => __('Keep in mind that URL Referrer discovery is not always 100% accurate. Some servers may use proxies that strip this information out and it can be easily forged.', 'fpf-framework'),
			'FPF_SELECT_A_DAY' => __('Select a day', 'fpf-framework'),
			'FPF_SELECT_A_MONTH' => __('Select a month', 'fpf-framework'),
			'FPF_MENU' => __('Menu', 'fpf-framework'),
			'FPF_MENU_ITEM' => __('Menu Item', 'fpf-framework'),
			'FPF_MENU_ITEM_HINT' => __('Start searching for a menu item', 'fpf-framework'),
			'FPF_MENU_INCLUDE_CHILD' => __('Also on child items', 'fpf-framework'),
			'FPF_MENU_INCLUDE_CHILD_DESC' => __('Set whether to include child items.', 'fpf-framework'),
			'FPF_LANGUAGE_HINT' => __('Start searching for a language', 'fpf-framework'),
			'FPF_WPML_LANGUAGE' => __('WPML Language', 'fpf-framework'),
			'FPF_POST_CATEGORY_HINT' => __('Start searching for a post category', 'fpf-framework'),
			'FPF_CPTS_HINT' => __('Start searching for a custom post type', 'fpf-framework'),
			'FPF_PAGE_VIEWS' => __('Page Views', 'fpf-framework'),
			'FPF_CONDITION' => __('Condition', 'fpf-framework'),
			'FPF_FIREBOX_SUBMITTED_FORM' => __('FireBox - Submitted Form', 'fpf-framework'),
			'FPF_FIREBOX_FORM' => __('FireBox Form', 'fpf-framework'),
			'FPF_SELECT_A_FIREBOX_FORM_HINT' => __('Start searching for a FireBox form', 'fpf-framework'),
			'FPF_NEW_RETURNING_VISITOR' => __('New/Returning Visitor', 'fpf-framework'),
			'FPF_VISITOR_IS_NEW' => __('Visitor is new', 'fpf-framework'),
			'FPF_VISITOR_IS_RETURNING' => __('Visitor is returning', 'fpf-framework'),
			'FPF_TEXT' => __('Text', 'fpf-framework'),
			'FPF_BACKGROUND_COLOR' => __('Background Color', 'fpf-framework'),
			'FPF_WOOCOMMERCE' => __('WooCommerce', 'fpf-framework'),
			'FPF_WOOCOMMERCE_PRODUCTS_IN_CART' => __('WooCommerce Products in Cart', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CART_ITEMS_COUNT' => __('WooCommerce Cart Items Count', 'fpf-framework'),
			'FPF_WOOCOMMERCE_AMOUNT_IN_CART' => __('WooCommerce Amount in Cart', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CURRENT_PRODUCT' => __('WooCommerce Current Product', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CURRENT_PRODUCT_CATEGORY' => __('WooCommerce Current Product Category', 'fpf-framework'),
			'FPF_SELECT_WOOCOMMERCE_CATEGORIES' => __('Select WooCommerce Categories...', 'fpf-framework'),
			'FPF_TARGET_VISITORS_WITH_SPECIFIC_PRODUCTS_ON_THEIR_CART' => __('Target visitors that have added specific products to their cart.', 'fpf-framework'),
			'FPF_PRODUCTS_IN_CART' => __('Products in Cart', 'fpf-framework'),
			'FPF_CART_ITEMS_COUNT' => __('Cart Items Count', 'fpf-framework'),
			'FPF_AMOUNT_IN_CART' => __('Amount in Cart', 'fpf-framework'),
			'FPF_EXCLUDE_SHIPPING_COST' => __('Exclude Shipping Cost', 'fpf-framework'),
			'FPF_SELECT_WOOCOMMERCE_PRODUCTS' => __('Select WooCommerce Products...', 'fpf-framework'),
			'FPF_EDD' => __('Easy Digital Downloads', 'fpf-framework'),
			'FPF_EDD_PRODUCTS_IN_CART' => __('Easy Digital Downloads Products in Cart', 'fpf-framework'),
			'FPF_EDD_CART_ITEMS_COUNT' => __('Easy Digital Downloads Cart Items Count', 'fpf-framework'),
			'FPF_EDD_AMOUNT_IN_CART' => __('Easy Digital Downloads Amount in Cart', 'fpf-framework'),
			'FPF_EDD_CURRENT_PRODUCT' => __('Easy Digital Downloads Current Product', 'fpf-framework'),
			'FPF_EDD_CURRENT_PRODUCT_CATEGORY' => __('Easy Digital Downloads Current Product Category', 'fpf-framework'),
			'FPF_SELECT_EDD_PRODUCTS' => __('Select Easy Digital Downloads Products...', 'fpf-framework'),
			'FPF_PRODUCT' => __('Product', 'fpf-framework'),
			'FPF_SELECT_EDD_CATEGORIES' => __('Select Easy Digital Downloads Categories...', 'fpf-framework'),
			'FPF_SEARCH_DOWNLOAD' => __('Search download...', 'fpf-framework'),
			'FPF_CART' => __('Cart', 'fpf-framework'),
			'FPF_AMOUNT_LC' => __('amount', 'fpf-framework'),
			'FPF_CART_AMOUNT' => __('Cart amount', 'fpf-framework'),
			'FPF_SET_QUANTITY_LC' => __('set quantity', 'fpf-framework'),
			'FPF_IS_ANY_LC' => __('is any', 'fpf-framework'),
			'FPF_TOTAL_LC' => __('total', 'fpf-framework'),
			'FPF_SUBTOTAL_LC' => __('subtotal', 'fpf-framework'),
			'FPF_CURRENT_PRODUCT_PRICE' => __('Current Product Price', 'fpf-framework'),
			'FPF_EDD_CURRENT_PRODUCT_PRICE' => __('Easy Digital Downloads Current Product Price', 'fpf-framework'),
			'FPF_EDD_CURRENT_PRODUCT_STOCK' => __('Easy Digital Downloads Current Product Stock', 'fpf-framework'),
			'FPF_CURRENT_PRODUCT_STOCK' => __('Current Product Stock', 'fpf-framework'),
			'FPF_EDD_PURCHASED_PRODUCT' => __('Easy Digital Downloads Purchased Product', 'fpf-framework'),
			'FPF_EDD_LAST_PURCHASED_DATE' => __('Easy Digital Downloads Last Purchased Date', 'fpf-framework'),
			'FPF_LAST_PURCHASED_DATE' => __('Last Purchased Date', 'fpf-framework'),
			'FPF_LAST_PURCHASED_DATE2' => __('Last Purchased Date 2', 'fpf-framework'),
			'FPF_IS_IN_THE_LAST_X_DAYS' => __('Is in the last X days', 'fpf-framework'),
			'FPF_IS_BEFORE' => __('Is before', 'fpf-framework'),
			'FPF_IS_AFTER' => __('Is after', 'fpf-framework'),
			'FPF_WEEKS' => __('Weeks', 'fpf-framework'),
			'FPF_MONTHS' => __('Months', 'fpf-framework'),
			'FPF_STOCK_LC' => __('stock', 'fpf-framework'),
			'FPF_TOTAL_SPEND' => __('Total Spend', 'fpf-framework'),
			'FPF_EDD_TOTAL_SPEND' => __('Easy Digital Downloads Total Spend', 'fpf-framework'),
			'FPF_HOMEPAGE' => __('Homepage', 'fpf-framework'),
			'FPF_VISITOR_IS' => __('Visitor is', 'fpf-framework'),
			'FPF_ON_HOMEPAGE' => __('On homepage', 'fpf-framework'),
			'FPF_NOT_ON_HOMEPAGE' => __('Not on homepage', 'fpf-framework'),
			'FPF_WOOCOMMERCE_PURCHASED_PRODUCT' => __('WooCommerce Purchased Product', 'fpf-framework'),
			'FPF_WOOCOMMERCE_LAST_PURCHASED_DATE' => __('WooCommerce Last Purchased Date', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CURRENT_PRODUCT_PRICE' => __('WooCommerce Current Product Price', 'fpf-framework'),
			'FPF_WOOCOMMERCE_TOTAL_SPEND' => __('WooCommerce Total Spend', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CURRENT_PRODUCT_STOCK' => __('WooCommerce Current Product Stock', 'fpf-framework'),
			'FPF_WOOCOMMERCE_CATEGORY' => __('WooCommerce Category', 'fpf-framework'),
			'FPF_EDD_CATEGORY' => __('Easy Digital Downloads Category', 'fpf-framework'),
			'FPF_BLANK_TEMPLATE' => __('Blank Template', 'fpf-framework'),
			'FPF_MINE' => __('Mine', 'fpf-framework'),
			'FPF_DRAFTS' => __('Drafts', 'fpf-framework'),
			'FPF_UPGRADE_TO_PRO_VERSION' => __('Awesome! Only one step left. Click on the button below to complete the upgrade to the Pro version.', 'fpf-framework'),
			'FPF_OPEN_EVENT' => __('Open', 'fpf-framework'),
			'FPF_CLOSE_EVENT' => __('Close', 'fpf-framework'),
			'FPF_DARK_MODE' => __('Dark mode', 'fpf-framework'),
			'FPF_VERSION' => __('Version', 'fpf-framework'),
			'FPF_PLUGIN_OUDATED_PLEASE_UPDATE' => __('%s is outdated. Please update.', 'fpf-framework'),
			'FPF_PLUGIN_IS_UP_TO_DATE' => __('%s is up to date.', 'fpf-framework'),
			'FPF_UPGRADE_NOW_20_OFF' => __('Upgrade now for 20% OFF and unlock all Pro features.', 'fpf-framework'),
			'FPF_GIVE_FEEDBACK' => __('Give feedback', 'fpf-framework'),
			'FPF_WHATS_NEW' => __('What\'s New', 'fpf-framework'),
			'FPF_FLAG' => __('Flag', 'fpf-framework'),
			'FPF_N/A' => __('n/a', 'fpf-framework'),
			'FPF_S' => __('s', 'fpf-framework'),
			'FPF_ACTIVATE' => __('Activate', 'fpf-framework'),
			'FPF_DEACTIVATE' => __('Deactivate', 'fpf-framework'),
			'FPF_EDIT' => __('Edit', 'fpf-framework'),
			'FPF_TOP_CENTER' => __('Top Center', 'fpf-framework'),
			'FPF_MIDDLE_LEFT' => __('Middle Left', 'fpf-framework'),
			'FPF_CENTER' => __('Center', 'fpf-framework'),
			'FPF_MIDDLE_RIGHT' => __('Middle Right', 'fpf-framework'),
			'FPF_BOTTOM_CENTER' => __('Bottom Center', 'fpf-framework'),
			'FPF_TO' => __('to', 'fpf-framework'),
			'FPF_UPGRADE_NOW' => __('Upgrade Now', 'fpf-framework'),
			'FPF_CONVERSION_EVENT' => __('Conversion', 'fpf-framework'),
			'FPF_TAG_GEOCOUNTRY' => __('Visitor Country', 'fpf-framework'),
			'FPF_TAG_GEOCOUNTRYCODE' => __('Visitor Country Code', 'fpf-framework'),
			'FPF_TAG_GEOCITY' => __('Visitor City', 'fpf-framework'),
			'FPF_TAG_GEOREGION' => __('Visitor Region', 'fpf-framework'),
			'FPF_TAG_GEOLOCATION' => __('Visitor Location Address', 'fpf-framework'),
			'FPF_TAG_COOKIEKEY' => __('Cookie Name Value', 'fpf-framework'),
			'FPF_TAG_POSTKEY' => __('POST Superglobal Value', 'fpf-framework'),
			'FPF_TAG_QUERYSTRINGKEY' => __('Query String Parameter Value', 'fpf-framework'),
		];
	}
}